import numpy as np
import matplotlib.pyplot as plt
from scipy.optimize import brentq

# Initial coordinates
X_0 = 5.0
Y_0 = 4.0

# Define the function
def F_psi(psi):
    return 2 * psi - np.sin(2 * psi) - (X_0 / Y_0) * (1 - np.cos(2 * psi))

# Plot F_psi to find a good root range
Psi = np.linspace(0.01, 2 * np.pi, 1000)
plt.figure(10)
plt.plot(Psi, F_psi(Psi))
plt.axhline(0, color='k', linestyle='--')
plt.title("Root of F(ψ)")
plt.xlabel("ψ (rad)")
plt.ylabel("F(ψ)")
plt.grid(True)
plt.show()

# Use brentq to safely find a root in a valid interval
Psi_0 = brentq(F_psi, 0.1, 3.0)  # This avoids root at zero
print(f"Psi_0 = {Psi_0:.4f} rad")

# Calculate constant
Const = (1 - np.cos(2 * Psi_0)) / Y_0

# Generate Brachistochrone curve
Psi_vals = np.linspace(0, Psi_0, 1000)
X_vals = (2 * Psi_vals - np.sin(2 * Psi_vals)) / Const
Y_vals = (1 - np.cos(2 * Psi_vals)) / Const

# Plot like MATLAB
plt.figure(20)
plt.plot(X_vals, Y_vals)
plt.gca().invert_yaxis()
plt.axis('equal')
plt.title("Brachistochrone Curve")
plt.xlabel("X")
plt.ylabel("Y (flipped)")
plt.grid(True)
plt.show()
