# Friction_linkage_response.py (Cleaned Up Version Matching MATLAB Console Behavior)

import numpy as np
from scipy.integrate import solve_ivp
import matplotlib.pyplot as plt
from scipy.optimize import fsolve
import warnings

# -----------------------------
# User Inputs
# Define system parameters here: These ones are varied in the discussion in the book:
# -----------------------------
mu_k = 0 # Friction Coefficient
# mu_k = 0.5 # Friction Coefficient
# Applied Torque Gamma
Gam = 1000
# Gam = 3000
# Gam = 1741.2991  # Use with mu_k=0
# Gam = 1741.2992 
# Gam = 2165.7  # Use with mu_k = 0.5
# Gam = 2165.72  # Use with mu_k=0.5
# Largest torque for return to theta = 0 is Gam = 1741.299 N-m
# Gam = float(input("Enter the amplitude of the torque (Gam): "))
t_max=2 # use t_max=0.5 for Gam=3000

# -----------------------------
# System Parameters
# -----------------------------
L = 0.5 # m
L_0 = 3 * L
beta_L = L_0 / L
m_AB = 6 # kg
m_BC = 24 # kg
k_stiff = 1e4

# -----------------------------
# Initial Conditions and Time
# -----------------------------
theta_0 = 0 * np.pi / 180
alpha_0 = np.sqrt(5 - 4 * np.cos(theta_0))
phi_0 = np.arcsin(np.sin(theta_0) / alpha_0)
q_0 = np.array([theta_0, phi_0])
q_dot_0 = np.array([0.0, 0.0])
Z_0 = np.concatenate([q_0, q_dot_0])

Omega = np.sqrt(k_stiff / (m_AB + m_BC))
T_nat = np.sqrt(2 * np.pi / Omega)
delta_t = T_nat / 1000
max_steps = int(np.ceil(t_max / delta_t))
t_vals = np.linspace(0, max_steps * delta_t, max_steps + 1)

q_vals = np.zeros((2, max_steps + 1))
q_dot_vals = np.zeros((2, max_steps + 1))
q_2dot_vals = np.zeros((2, max_steps + 1))
N_D_vals = np.zeros(max_steps + 1)
q_vals[:, 0] = q_0
q_dot_vals[:, 0] = q_dot_0

# -----------------------------
# Eval_N_d Function
# -----------------------------
def Eval_N_d(Z, t_f, N_D_prev):
    q = Z[:2]; q_dot = Z[2:]
    M_mat = np.array([[(m_AB / 3 + m_BC) * L**2, -2 * m_BC * np.cos(q[0] + q[1]) * L**2],
                      [-2 * m_BC * np.cos(q[0] + q[1]) * L**2, (16 / 3) * m_BC * L**2]])
    alpha = np.sqrt(5 - 4 * np.cos(q[0]))
    Gamma_spr_2 = -2 * k_stiff * L**2 * np.sin(q[0]) * (1 - (4 - beta_L) / alpha)
    FF = np.array([Gam - 2 * m_BC * L**2 * np.sin(q[0] + q[1]) * q_dot[1]**2 + Gamma_spr_2,
                   -2 * m_BC * L**2 * np.sin(q[0] + q[1]) * q_dot[0]**2])
    sigma = L * np.array([np.cos(q[0] + q[1]), -alpha])
    gamma = np.array([-mu_k * 2 * L * np.abs(np.sin(q[1])) * np.sign(q_dot[0]), 0])
    a = np.array([[np.cos(q[0] + q[1]), np.cos(q[0] + q[1]) - 2 * np.cos(q[1])]])
    da_dt = np.array([[-(q_dot[0] + q_dot[1]) * np.sin(q[0] + q[1]),
                       -(q_dot[0] + q_dot[1]) * np.sin(q[0] + q[1]) + 2 * q_dot[1] * np.sin(q[1])]])
    RHS = np.concatenate((FF, da_dt @ q_dot))
    LHS = np.block([[M_mat, -(sigma[:, None] + gamma[:, None] * np.sign(N_D_prev))], [-a, np.zeros((1, 1))]])
    X = np.linalg.solve(LHS, RHS)
    return X[2], X[:2]

# -----------------------------
# G_vec Function
# -----------------------------
def G_vec(t, Z, N_D):
    _, q_2dot = Eval_N_d(Z, t, N_D)
    return np.concatenate((Z[2:], q_2dot))

# -----------------------------
# Run Simulation
# -----------------------------
Z_prev = Z_0
N_D = 0
q_2dot_vals[:, 0] = G_vec(0, Z_0, N_D)[2:]
for m in range(1, max_steps + 1):
    sol = solve_ivp(lambda t, Z: G_vec(t, Z, N_D), [t_vals[m - 1], t_vals[m]], Z_prev, method='RK45', rtol=1e-8, atol=1e-8)
    Z_prev = sol.y[:, -1]
    N_D, q_2dot = Eval_N_d(Z_prev, t_vals[m], N_D)
    q_vals[:, m], q_dot_vals[:, m], q_2dot_vals[:, m], N_D_vals[m] = Z_prev[:2], Z_prev[2:], q_2dot, N_D

# -----------------------------
# Post-Processing and Plotting
# -----------------------------
time_ms = t_vals * 1000
theta = q_vals[0]; phi = q_vals[1]
theta_dot = q_dot_vals[0]; phi_dot = q_dot_vals[1]
F_spr = k_stiff * L * (2 - np.sqrt(5 - 4 * np.cos(theta)))

print("\nMax configuration constraint error =", np.max(np.abs(np.sin(theta + phi) - 2 * np.sin(phi))))

min_phi = np.min(np.abs(phi[200:])) * 180 / np.pi
min_theta = np.min(np.abs(theta[200:])) * 180 / np.pi
max_phi = np.max(np.abs(phi[200:])) * 180 / np.pi
max_theta = np.max(np.abs(theta[200:])) * 180 / np.pi
print(f"\nmin theta = {min_theta:.4f}, min phi = {min_phi:.4f}")
print(f"max theta = {max_theta:.4f}, max phi = {max_phi:.4f}")

fignum = 1;

plt.figure(fignum)
plt.subplot(3, 1, 1)
plt.plot(time_ms, np.mod(theta * 180 / np.pi + 180, 360) - 180, 'r-', label=r'$\theta$')
plt.plot(time_ms, phi * 180 / np.pi, 'b-o', label=r'$\phi$', markersize=2, markevery=100)
plt.ylabel('Angle (deg)'); plt.legend(); plt.title(rf"$\Gamma$={Gam}, k={k_stiff}, $\mu$={mu_k}, max($\phi$)={max_phi:.4f}")

plt.subplot(3, 1, 2)
plt.plot(time_ms, theta_dot, 'r-', label=r'$d\theta/dt$')
plt.plot(time_ms, phi_dot, 'b--o', label=r'$d\phi/dt$', markersize=2, markevery=100)
plt.xlabel('Time (ms)'); plt.ylabel('ang vel (rad/s)'); plt.legend()

plt.subplot(3, 1, 3)
plt.plot(time_ms, N_D_vals, 'r-', label=r'$N_D$')
plt.plot(time_ms, F_spr, 'b--o', label=r'$k\Delta', markersize=2, markevery=100)
plt.xlabel('Time (ms)'); plt.ylabel('Force (N)'); plt.legend()
plt.tight_layout(); plt.show()

# -----------------------------
# Static Torque & Energy Plot
# -----------------------------
thetas = np.radians(np.arange(0, 181, 0.2))
alphas = np.sqrt(5 - 4 * np.cos(thetas))
deltas = (1 - alphas) * L
V = 0.5 * k_stiff * deltas**2
Gamma_static = (2 / alphas) * k_stiff * L * np.abs(deltas) * np.sin(thetas)

with warnings.catch_warnings():
    warnings.simplefilter("ignore")
    V_over_theta = np.divide(V, thetas, out=np.zeros_like(V), where=thetas != 0)

plt.figure(fignum + 10)
plt.plot(np.degrees(thetas), V_over_theta, 'r', label='V/\u03b8')
plt.plot(np.degrees(thetas), Gamma_static, 'b--', label='\u0393_static')
plt.xlabel('\u03b8 (deg)'); plt.ylabel('Value'); plt.legend(); plt.grid(True); plt.title("Static Torque and Energy Analysis")
plt.tight_layout(); plt.show()

# -----------------------------
# Print Locking Torque & Angle
# -----------------------------
alpha_func = lambda theta: np.sqrt(5 - 4 * np.cos(theta))
delta_func = lambda theta: (1 - alpha_func(theta)) * L
V_func = lambda theta: 0.5 * k_stiff * delta_func(theta)**2
Gam_stat_func = lambda theta: (2 / alpha_func(theta)) * k_stiff * L * np.abs(delta_func(theta)) * np.sin(theta)
F_lock = lambda theta: Gam_stat_func(theta) - V_func(theta) / theta

th_lock = fsolve(F_lock, np.radians(140))[0]
Gam_lock = Gam_stat_func(th_lock)

print(f"\nth_lock = {th_lock:.4f}")
print(f"\nGam_lock ={Gam_lock:.4e}\n")
