# Converted from MATLAB: Response_rack_gear.m and G_vec_rack.m
# This Python script computes and animates the response of a holonomic system

import numpy as np
import matplotlib.pyplot as plt
from scipy.integrate import solve_ivp

# System parameters
class Params:
    def __init__(self):
        self.m_r = 20
        self.m_A = 10
        self.m_s = 2
        self.kappa = 0.150
        self.radius = 0.200
        self.e_offset = 0.175
        self.k_stiff = 20
        self.freq = 4
        self.F_mag = 25

p = Params()
Ig = p.m_A * p.kappa**2

# Initial conditions
q_0 = np.array([2 * p.radius, 0])
q_dot_0 = np.array([0, 0])
Z_0 = np.concatenate((q_0, q_dot_0))

# Time setup
T_force = 2 * np.pi / p.freq
T_max = 16 * T_force
N_intervals = 16

delta_t = T_force / N_intervals
t_out = np.arange(0, T_max + delta_t, delta_t)

# Function to evaluate dZ/dt
def G_vec_rack(t, Z, p):
    q = Z[:2]
    q_dot = Z[2:]
    Force = p.F_mag * np.sin(p.freq * t)
    Q = [Force,
         (p.radius + p.e_offset * np.cos(q[1])) * Force]
    FF1 = Q[0] + (p.m_A + p.m_r) * p.e_offset * np.sin(q[1]) * q_dot[1]**2 - p.k_stiff * (q[0] - 2 * p.radius)
    FF2 = Q[1] + ((p.m_A + p.m_r) * p.e_offset**2 * 0.5 * np.sin(2 * q[1]) + p.m_r * p.radius * p.e_offset * np.sin(q[1])) * q_dot[1]**2
    FF = np.array([FF1, FF2])

    M11 = p.m_A + p.m_r + p.m_s
    M12 = p.m_r * p.radius + (p.m_A + p.m_r) * p.e_offset * np.cos(q[1])
    M21 = M12
    M22 = (p.m_A * p.kappa**2 +
           p.m_A * p.e_offset**2 * np.cos(q[1])**2 +
           p.m_r * (p.radius + p.e_offset * np.cos(q[1]))**2)
    M = np.array([[M11, M12], [M21, M22]])
    q_ddot = np.linalg.solve(M, FF)
    return np.concatenate((q_dot, q_ddot))

# Solve the ODE
def dynamics(t, Z):
    return G_vec_rack(t, Z, p)

sol = solve_ivp(dynamics, [t_out[0], t_out[-1]], Z_0, t_eval=t_out, rtol=1e-10, atol=1e-10)
t_ode = sol.t
Z_ode = sol.y.T

x = Z_ode[:, 0]
theta = Z_ode[:, 1]
x_rack = x + p.e_offset * np.sin(theta) + p.radius * theta
y_B = p.e_offset * np.cos(theta)
t_nondim = t_ode * p.freq / (2 * np.pi)

# Plot response (Figure 10)
plt.figure(10)
ax1 = plt.gca()
ax1.plot(t_nondim, x, 'b--', label='x')
ax1.plot(t_nondim, x_rack, 'r-', label='x_r')
ax1.set_xlabel('t/T_period')
ax1.set_ylabel('Position (m)', color='b')
ax1.tick_params(axis='y', labelcolor='b')
ax1.set_xlim([0, 16])
ax1.set_ylim([-0.8, 1.6])

ax2 = ax1.twinx()
ax2.plot(t_nondim, np.rad2deg(theta), 'k:', label='θ')
ax2.set_ylabel('angle (deg)', color='k')
ax2.tick_params(axis='y', labelcolor='k')
ax2.set_ylim([-180, 360])

lines1, labels1 = ax1.get_legend_handles_labels()
lines2, labels2 = ax2.get_legend_handles_labels()
ax1.legend(lines1 + lines2, labels1 + labels2, loc='upper right')
ax1.grid(True)

# Derivatives
Force_ext = p.F_mag * np.sin(p.freq * t_ode)
x_dot = np.gradient(x, delta_t)
theta_dot = np.gradient(theta, delta_t)
x_2dot = np.gradient(x_dot, delta_t)
theta_2dot = np.gradient(theta_dot, delta_t)

x_A_2dot = x_2dot + p.e_offset * theta_2dot * np.cos(theta) - p.e_offset * theta_dot**2 * np.sin(theta)
x_rack_2dot = x_A_2dot + p.radius * theta_2dot

Res_slider = p.m_s * x_2dot
Res_gear = p.m_A * x_A_2dot
Res_rack = p.m_r * x_rack_2dot
Spring_force = -p.k_stiff * (x - 2 * p.radius)
Pin_force = Res_slider - Spring_force
Rack_force = -Res_rack + Force_ext

Moment_pin = Pin_force * p.e_offset * np.cos(theta)
Moment_rack = Rack_force * p.radius
Moment_gear = Ig * theta_2dot

# Figure 30
plt.figure(30)
plt.subplot(2,1,1)
plt.plot(t_nondim, -Rack_force, 'r--', label='-F_r')
plt.plot(t_nondim, Force_ext, 'b:', label='Excitation')
plt.plot(t_nondim, Res_rack, 'k-', label='m_r*d²x_r/dt²')
plt.xlabel('t/T_F')
plt.ylabel('Forces (N)')
plt.xlim([0, 4])
plt.ylim([-30, 40])
plt.grid(True)
plt.legend()

plt.subplot(2,1,2)
plt.plot(t_nondim, Pin_force, 'r:', label='F_B')
plt.plot(t_nondim, Spring_force, 'b--', label='-k*x')
plt.plot(t_nondim, Res_slider, 'k-', label='m_s*d²x/dt²')
plt.xlabel('t/T_F')
plt.ylabel('Forces (N)')
plt.xlim([0, 4])
plt.ylim([-6, 6])
plt.grid(True)
plt.legend()

# Figure 40
plt.figure(40)
plt.subplot(2,1,1)
plt.plot(t_nondim, Rack_force, 'r:', label='F_r')
plt.plot(t_nondim, -Pin_force, 'b--', label='-F_B')
plt.plot(t_nondim, Res_gear, 'k-', label='m_A*a_A')
plt.xlabel('t/T_F')
plt.ylabel('Forces (N)')
plt.xlim([0, 8])
plt.ylim([-12, 12])
plt.grid(True)
plt.legend()

plt.subplot(2,1,2)
plt.plot(t_nondim, Moment_pin, 'r--', label='M_B')
plt.plot(t_nondim, Moment_rack, 'b:', label='M_r')
plt.plot(t_nondim, Moment_gear, 'k-', label='I_A*d²θ/dt²')
plt.xlabel('t/T_period')
plt.ylabel('Forces (N)')
plt.xlim([0, 4])
plt.ylim([-2, 2])
plt.grid(True)
plt.legend()

# Figure 20 - Static Snapshots
ths = np.radians(np.arange(0, 361, 1))
xg_pts = p.radius * np.sin(ths)
yg_pts = p.radius * (1 + np.cos(ths))

xs_pts = np.array([-0.3, 0.3, 0.3, 0.1, 0.1, -0.1, -0.1, -0.3, -0.3]) * p.radius
ys_pts = (np.array([0.1, 0.1, -0.1, -0.1, -2.2, -2.2, -0.1, -0.1, 0.1]) + 2.5) * p.radius

xr_pts = np.array([0, -4, -4, 0, 0])
yr_pts = np.array([0, 0, -0.1, -0.1, 0])

k_index = np.arange(0, min(65, len(t_ode)), 4)
N_col = 2
N_row = 3
N_plot = 0
fignum = 50

for idx, k in enumerate(k_index):
    xB = x[k]
    theta_k = theta[k]
    xg_center = x[k] + p.e_offset * np.sin(theta_k)
    xg = xg_center + xg_pts
    yg = yg_pts

    xs = xs_pts + x[k]
    ys = ys_pts
    xr = xr_pts + x_rack[k]
    yr = yr_pts
    yB = p.radius + p.e_offset * np.cos(theta_k)

    if N_plot == N_row * N_col:
        fignum += 10
        N_plot = 0
    N_plot += 1
    if N_plot == 1:
        plt.figure(fignum)

    plt.subplot(N_row, N_col, N_plot)
    plt.fill(xs, ys, color='blue', alpha=0.7)
    plt.fill(xr, yr, color='green', alpha=0.6)
    plt.fill(xg, yg, color='red', alpha=0.5)
    plt.plot(xB, yB, 'yo', markersize=4)
    plt.plot([x[0], x[0]], [0, 3.5 * p.radius], 'k--', linewidth=1)
    plt.text(-0.75, 0.25, f'{t_ode[k]/T_force:.2f}', fontsize=8)
    plt.axis('equal')
    plt.xlim([-2, 3])
    plt.ylim([0, 1.5])
    plt.xticks([])
    plt.yticks([])

plt.suptitle("Figure 20: Rack-Gear-Slider System Snapshots", fontsize=14)
plt.tight_layout()
plt.show()
