# Response_Unconstrained.py
# This script evaluates the response of a holonomic system
# described by unconstrained generalized coordinates.
#
# This program requires creation of a function G_vec
# that evaluates dZ/dt.
#
# In addition, system parameters and initial conditions
# for the generalized coordinates must be defined below,
# along with the maximum time for integration and number of time steps.

import numpy as np
from scipy.integrate import solve_ivp

# Define system parameters here:
param1 = ...  # Example: mass, stiffness, geometry, etc.
param2 = ...

# Set initial conditions:
t_0 = ...        # Initial time
q_0 = ...        # Initial generalized coordinates (as column vector or 1D array)
q_dot_0 = ...    # Initial generalized velocities (same format as q_0)
N = len(q_0)     # Number of generalized coordinates

# Time-stepping setup
N_steps = ...  # Total number of time steps for response output
t_min = t_0
t_max = ...
delta_t = (t_max - t_min) / N_steps
t_vals = np.array([i * delta_t for i in range(N_steps)])  # Time vector

# Allocate memory for solution arrays
q_vals = np.zeros((N, N_steps))         # Stores q values column-wise
q_dot_vals = np.zeros((N, N_steps))     # Stores q_dot values column-wise

# Save initial values
q_vals[:, 0] = q_0
q_dot_vals[:, 0] = q_dot_0
Z_prev = np.concatenate((q_0, q_dot_0))  # Combine into state-space vector

# Begin time stepping loop
for m in range(1, N_steps):
    # Use lambda to treat G_vec as function of t and Z only
    # Default solver tolerances used unless specified
    sol = solve_ivp(
        fun=lambda t, Z: G_vec(Z, t, param1, param2),
        t_span=(t_vals[m - 1], t_vals[m]),
        y0=Z_prev,
        method='RK45'
        # Optional: add rtol=1e-8, atol=1e-8 here
    )

    # solve_ivp returns a solution object where `sol.y` stores rows of Z
    # at each integration time stored in `sol.t`
    Z_last = sol.y[:, -1]          # Last column = solution at t_vals[m]
    Z_prev = Z_last                # For next step
    q_vals[:, m] = Z_last[:N]
    q_dot_vals[:, m] = Z_last[N:]
