import numpy as np
import matplotlib.pyplot as plt

# Parameters
beta_deg = 75
beta = np.radians(beta_deg)
mu = 0.25
theta_deg = beta_deg - np.arange(0, 75.5, 0.5)
theta_deg[0] = beta_deg - 0.0001

sig_vals = np.array([[1, 1, -1, -1],
                     [1, -1, 1, -1]])

F = np.zeros((4, len(theta_deg)))
N_A = np.zeros_like(F)
N_B = np.zeros_like(F)

# Loop over sign combinations
for n in range(4):
    sigma_A = sig_vals[0, n]
    sigma_B = sig_vals[1, n]
    for j, theta_d in enumerate(theta_deg):
        theta = np.radians(theta_d)
        D = np.array([
            [1, -mu, sigma_B * np.sin(beta) - mu * np.cos(beta)],
            [0, sigma_A, sigma_B * np.cos(beta) + mu * np.sin(beta)],
            [np.sin(theta),
             sigma_A * np.cos(theta) - mu * np.sin(theta),
             -sigma_B * np.cos(beta - theta) - mu * np.sin(beta - theta)]
        ])
        K = (np.sin(beta)**2 / np.cos(beta - theta)**3) * np.array([
            0.5 * np.cos(beta),
            -0.5 * np.sin(beta),
            -np.sin(beta - theta) / 6
        ])
        X = np.linalg.solve(D, K)
        F[n, j] = X[0]
        N_A[n, j] = X[1]
        N_B[n, j] = X[2]

# === Plot Figure 10 Equivalent ===
fig1, axs1 = plt.subplots(1, 2, figsize=(12, 5))
styles = ['k-', 'b--', 'g-.', 'r:']
labels = [r'$\sigma_A=1, \sigma_B=1$', r'$\sigma_A=1, \sigma_B=-1$',
          r'$\sigma_A=-1, \sigma_B=1$', r'$\sigma_A=-1, \sigma_B=-1$']

for n in range(4):
    axs1[0].plot(theta_deg, N_A[n, :], styles[n], linewidth=1, label=labels[n])
    axs1[1].plot(theta_deg, N_B[n, :], styles[n], linewidth=1, label=labels[n])

axs1[0].set_xlim([0, 75])
axs1[0].invert_xaxis()
axs1[0].legend()
axs1[0].set_title("N_A vs Theta (deg)")

axs1[1].set_xlim([0, 75])
axs1[1].set_ylim([-100, 400])
axs1[1].invert_xaxis()
axs1[1].legend()
axs1[1].set_title("N_B vs Theta (deg)")

plt.tight_layout()

# === Plot Figure 20 Equivalent ===
theta = np.radians(theta_deg)
vt_L = np.sin(beta - theta) / np.sin(beta)
sigma_A = -1
sigma_B = -1

FF = []
NN_A = []
NN_B = []

for theta_n in theta:
    D = np.array([
        [1, -mu, sigma_B * np.sin(beta) - mu * np.cos(beta)],
        [0, sigma_A, sigma_B * np.cos(beta) + mu * np.sin(beta)],
        [np.sin(theta_n),
         sigma_A * np.cos(theta_n) - mu * np.sin(theta_n),
         -sigma_B * np.cos(beta - theta_n) - mu * np.sin(beta - theta_n)]
    ])
    K = (np.sin(beta)**2 / np.cos(beta - theta_n)**3) * np.array([
        0.5 * np.cos(beta),
        -0.5 * np.sin(beta),
        -np.sin(beta - theta_n) / 6
    ])
    Y = np.linalg.solve(D, K)
    FF.append(Y[0])
    NN_A.append(Y[1])
    NN_B.append(Y[2])

fig2, ax2 = plt.subplots(figsize=(6, 5))
ax2.semilogy(vt_L, FF, 'k-', label='F')
ax2.semilogy(vt_L, NN_A, 'r--', label=r'$N_A$')
ax2.semilogy(vt_L, NN_B, 'b:', label=r'$N_B$')
ax2.set_ylim([1e-4, 1e4])
ax2.set_yticks([1e-4, 1e-2, 1, 1e2, 1e4])
ax2.set_xticks(np.arange(0, 1.1, 0.2))
ax2.set_xlabel('vt/L')
ax2.legend(loc='upper left')
ax2.set_title("Log Plot of Forces vs vt/L")

plt.tight_layout()
plt.show()
