'''
Example 4.11, Simulation of Squeeze-rack and Gear
'''

import numpy as np
import matplotlib.pyplot as plt
from scipy.integrate import quad

# Constants
v = 1
R = 1
v_fac = v / R
a_fac = v**2 / R

theta = np.radians(np.arange(1, 90))
L = R / np.tan(theta / 2)

v_A = v / (1 + np.cos(theta))
om_A = -v_fac * np.cos(theta) / (1 + np.cos(theta))
theta_dot = v_fac * (1 - np.cos(theta)) / (1 + np.cos(theta))
F_theta = np.sin(theta)**3 / (1 + np.cos(theta))**4

v_A_dot = v * F_theta
om_A_dot = v_fac**2 * F_theta
theta_2dot = 2 * v_fac**2 * F_theta

v_B_gear = []
v_E_gear = []
v_E_CD = []
a_B_gear = []
a_E_gear = []
a_E_CD = []

for i, th_n in enumerate(theta):
    v_A_XYZ = v_A[i] * np.array([-1, 0])
    v_B = v_A_XYZ + om_A[i] * R * np.array([1, 0])
    v_B_gear.append(v_B)

    a_A_XYZ = v_A_dot[i] * np.array([-1, 0])
    a_B = a_A_XYZ + om_A_dot[i] * R * np.array([1, 0]) + om_A[i]**2 * R * np.array([0, 1])
    a_B_gear.append(a_B)

    v_A_xyz = v_A[i] * np.array([-np.cos(th_n), np.sin(th_n)])
    v_E = v_A_xyz + om_A[i] * R * np.array([-1, 0])
    v_CD = L[i] * theta_dot[i] * np.array([0, 1])
    v_E_gear.append(v_E)
    v_E_CD.append(v_CD)

    a_A_xyz = v_A_dot[i] * np.array([-np.cos(th_n), np.sin(th_n)])
    a_E = a_A_xyz + om_A_dot[i] * R * np.array([-1, 0]) + om_A[i]**2 * R * np.array([0, -1])
    a_CD = L[i] * theta_2dot[i] * np.array([0, 1]) + L[i] * theta_dot[i]**2 * np.array([-1, 0])
    a_E_gear.append(a_E)
    a_E_CD.append(a_CD)

v_B_gear = np.array(v_B_gear).T
v_E_gear = np.array(v_E_gear).T
v_E_CD = np.array(v_E_CD).T
a_B_gear = np.array(a_B_gear).T
a_E_gear = np.array(a_E_gear).T
a_E_CD = np.array(a_E_CD).T

theta_deg = np.degrees(theta)

# Plotting
plt.figure(10)

plt.subplot(3, 2, 1)
plt.plot(theta_deg, v_B_gear[0], 'r-', label='x')
plt.plot(theta_deg, v_B_gear[1], 'b--', label='y')
plt.plot(theta_deg[::6], -1 * np.ones_like(theta_deg[::6]), 'ro', markersize=2)
plt.plot(theta_deg[::6], np.zeros_like(theta_deg[::6]), 'bd', markersize=2)
plt.ylim([-1.5, 0.5])
plt.xlim([0, 90])
plt.ylabel(r'$v_B$')
plt.legend()

plt.subplot(3, 2, 2)
plt.plot(theta_deg, v_E_gear[0], 'r-', label=r'$x_{gear}$')
plt.plot(theta_deg, v_E_CD[0], 'b--', label=r'$x_{CD}$')
plt.plot(theta_deg, v_E_gear[1], 'r-', label=r'$y_{gear}$')
plt.plot(theta_deg, v_E_CD[1], 'b--', label=r'$y_{CD}$')
plt.plot(theta_deg[::6], v_E_CD[0, ::6], 'ro', markersize=2)
plt.plot(theta_deg[::6], v_E_CD[1, ::6], 'bd', markersize=2)
plt.subplots_adjust(hspace=0.3,wspace=0.35,right=0.85)
plt.ylim([-0.5, 1])
plt.ylabel(r'$v_E$')
plt.legend(loc='center left', bbox_to_anchor=(1.0, 0.5))

plt.subplot(3, 2, 3)
plt.plot(theta_deg, a_B_gear[0], 'r-', label='x')
plt.plot(theta_deg, a_B_gear[1], 'b--', label='y')
plt.plot(theta_deg[::6], np.zeros_like(theta_deg[::6]), 'ro', markersize=2)
plt.plot(theta_deg[3::6], np.zeros_like(theta_deg[3::6]), 'bd', markersize=2)
plt.ylim([-0.1, 0.3])
plt.xlim([0, 90])
plt.ylabel(r'$a_B$')
plt.xlabel(r'$\theta$ (deg)')
plt.legend()

plt.subplot(3, 2, 4)
plt.plot(theta_deg, a_E_gear[0], 'r-', label=r'$x_{gear}$')
plt.plot(theta_deg, a_E_CD[0], 'ko', markersize=1.5, markevery=4, label=r'$x_{CD}$')
plt.plot(theta_deg, a_E_gear[1], 'b-', label=r'$y_{gear}$')
plt.plot(theta_deg, a_E_CD[1], 'b--', label=r'$y_{CD}$')
plt.xlim([0, 90])
plt.ylabel(r'$a_E$')
plt.xlabel(r'$\theta$ (deg)')
plt.legend(loc='center left', bbox_to_anchor=(1.0, 0.5))

# Time plot
theta_0 = 5
theta_t = np.radians(np.arange(theta_0, 90))
t = [0]

for i in range(1, len(theta_t)):
    integral, _ = quad(lambda theta: (1 + np.cos(theta)) / (1 - np.cos(theta)), theta_t[i-1], theta_t[i])
    t.append(t[-1] + integral)

t = np.array(t)
X_B = 1 / np.tan(theta_t / 2)

plt.figure(20)
plt.subplot(3, 2, 1)
plt.plot(t, np.degrees(theta_t), 'r', linewidth=1)
plt.xlabel('vt/R')
plt.ylabel(r'$\theta$ (deg)')
plt.ylim([theta_0, 90])
plt.yticks([5, 30, 60, 90])

plt.subplot(3, 2, 3)
plt.plot(X_B, t, 'r', linewidth=1)
plt.xlabel(r'$X_B$/R')
plt.ylabel('vt/R')

plt.tight_layout()
plt.show()
