'''
Calculations for Section 3.5 on influence of Earth's motion on gravity
'''
import numpy as np
import matplotlib.pyplot as plt

# === EARTH & ORBITAL CONSTANTS ===
T_day = 23 * 3600 + 56 * 60 + 4.06
omega_spin = 2 * np.pi / T_day

T_orbit = 365.256 * T_day
omega_orbit = 2 * np.pi / T_orbit
R_orbit = 149.6e6 * 1000
v_orbit = omega_orbit * R_orbit

T_sun_orbit = 230e6 * (365.25 * T_day)
omega_sun = 2 * np.pi / T_sun_orbit
v_sun_orbit = 720e3 * 1000
R_sun_orbit = v_sun_orbit / omega_sun

a_G_sun = omega_sun**2 * R_sun_orbit
a_G_earth = omega_orbit**2 * R_orbit

G = 6.6743e-11
M_sun = 1.9891e30
m_earth = 5.97219e24
R_e = 6371e3
m = 1

F_grav = G * m_earth * m / R_e**2
F_acc = m * R_e * omega_spin**2
frac = F_acc / F_grav

# === PRINTED OUTPUT ===
print(f"T_day = {T_day:.4e}")
print(f"omega_spin = {omega_spin:.4e}")
print(f"T_orbit = {T_orbit:.4e}")
print(f"omega_orbit = {omega_orbit:.4e}")
print(f"R_orbit = {R_orbit:.4e}")
print(f"v_orbit = {v_orbit:.4e}")
print(f"omega_sun = {omega_sun:.4e}")
print(f"v_sun_orbit = {v_sun_orbit:.4e}")
print(f"R_sun_orbit = {R_sun_orbit:.4e}")
print(f"a_G_sun = {a_G_sun:.4e}")
print(f"a_G_earth = {a_G_earth:.4f}")
print(f"F_grav = {F_grav:.4f}")
print(f"F_acc = {F_acc:.4f}")
print(f"frac = {frac:.4f}")

# === LATITUDE-BASED VARIATIONS ===
lambda_deg = np.arange(-90, 91)
lambda_rad = np.radians(lambda_deg)

mg_F_grav = 1 - frac * np.cos(lambda_rad)**2
chi = np.arctan(frac * np.sin(lambda_rad) * np.cos(lambda_rad))
chi_deg = np.degrees(chi)

# === PLOT 1: Gravity factor and angular deviation ===
plt.figure(10)
plt.subplot(3, 1, 1)
plt.plot(lambda_deg, mg_F_grav, '-r', label='mg/$F_{grav}$')
plt.legend()

ax2 = plt.gca().twinx()
ax2.plot(lambda_deg, chi_deg, ':b', label='$\chi$ (deg)')
plt.legend()
plt.subplot(3, 1, 1)
plt.xlabel(r'$\lambda$ (deg)')
plt.legend(loc='upper left')
ax2.legend(loc='upper right')

# === FREE-FALL DRIFT AT EQUATOR ===
lambda_f = 0
g = 9.807
H = 1000
y_f = -(1/3) * omega_spin * np.cos(lambda_f) * np.sqrt(32 * H**3 / g)
print(f"y_f = {y_f:.4f}")

# === PLOT 2: Coriolis-influenced trajectories ===
v_t = np.arange(0.25, 1.01, 0.01)
w_t = v_t / 2
psi = np.arange(-1, 1.01, 0.1) * np.pi
lambda_angle = np.pi / 3

plt.figure(20)
radius = max(v_t)
for psi_val in psi:
    x = (-radius + v_t) * np.cos(psi_val) + w_t * v_t * np.sin(lambda_angle) * np.sin(psi_val)
    y = (-radius + v_t) * np.sin(psi_val) - w_t * v_t * np.sin(lambda_angle) * np.cos(psi_val)
    plt.plot(x, y, '-k', linewidth=1)

plt.axis('equal')
plt.show()
