'''
Examples 5 and 6 in Chapter 3 - Arbitrary Rotation Axis
'''

import numpy as np

phi = -40 * np.pi / 180

L_Z_X, L_Z_Y = -0.4, 0.3
L_Z_Z = -np.sqrt(1 - L_Z_X**2 - L_Z_Y**2)
K_prime = np.array([L_Z_X, L_Z_Y, L_Z_Z])
print("K_prime =")
print(f"{K_prime[0]:8.4f}  {K_prime[1]:8.4f}  {K_prime[2]:8.4f}\n")

theta = np.arccos(L_Z_Z)
psi = np.arctan2(L_Z_Y / np.sin(theta), L_Z_X / np.sin(theta))
print("(theta,psi) (radians) =")
print(f"{theta:8.4f}  {psi:8.4f}\n")
print("(theta,psi) (degrees) =")
print(f"{np.degrees(theta):8.4f}  {np.degrees(psi):8.4f}\n")

R_psi = np.array([
    [np.cos(psi), np.sin(psi), 0],
    [-np.sin(psi), np.cos(psi), 0],
    [0, 0, 1]
])
R_theta = np.array([
    [np.cos(theta), 0, -np.sin(theta)],
    [0, 1, 0],
    [np.sin(theta), 0, np.cos(theta)]
])
R_prime = R_theta @ R_psi
print("R_prime =")
for row in R_prime: print(f"{row[0]:8.4f}  {row[1]:8.4f}  {row[2]:8.4f}")
print()

R_phi = np.array([
    [np.cos(phi), np.sin(phi), 0],
    [-np.sin(phi), np.cos(phi), 0],
    [0, 0, 1]
])
print("R_phi =")
for row in R_phi: print(f"{row[0]:8.4f}  {row[1]:8.4f}  {row[2]:8.4f}")
print()

R = R_prime.T @ R_phi @ R_prime
print("R =")
for row in R: print(f"{row[0]:8.4f}  {row[1]:8.4f}  {row[2]:8.4f}")
print()

print("Now work backward")
eigenvalues, eigenvectors = np.linalg.eig(R)

print("\neigenvectors =")
for i in range(3):
    for j in range(3):
        re, im = eigenvectors[i, j].real, eigenvectors[i, j].imag
        print(f"{re:7.4f} {'+' if im >= 0 else '-'} {abs(im):6.4f}i", end='   ')
    print()

print("\neigenvalues =")
for i in range(3):
    for j in range(3):
        re = eigenvalues[j].real if i == j else 0
        im = eigenvalues[j].imag if i == j else 0
        print(f"{re:7.4f} {'+' if im >= 0 else '-'} {abs(im):6.4f}i", end='   ')
    print()

idx = np.argmax(np.isclose(eigenvalues, 1.0))
e_phi = eigenvectors[:, idx].real
print("\ne_phi =")
for val in e_phi: print(f"{val:8.4f}")
print()

theta_eq = np.arccos(e_phi[2])
psi_eq = np.arctan2(e_phi[1] / np.sin(theta_eq), e_phi[0] / np.sin(theta_eq))
print(f"ans = {np.degrees(psi_eq):8.4f}  {np.degrees(theta_eq):8.4f}\n")

R_psi_eq = np.array([
    [np.cos(psi_eq), np.sin(psi_eq), 0],
    [-np.sin(psi_eq), np.cos(psi_eq), 0],
    [0, 0, 1]
])
R_theta_eq = np.array([
    [np.cos(theta_eq), 0, -np.sin(theta_eq)],
    [0, 1, 0],
    [np.sin(theta_eq), 0, np.cos(theta_eq)]
])
R_prime_eq = R_theta_eq @ R_psi_eq
print("R_prime_eq =")
for row in R_prime_eq: print(f"{row[0]:8.4f}  {row[1]:8.4f}  {row[2]:8.4f}")
print()

R_phi_eq = R_prime_eq @ R @ R_prime_eq.T
print("R_phi_eq =")
for row in R_phi_eq: print(f"{row[0]:8.4f}  {row[1]:8.4f}  {row[2]:8.4f}")
print()

phi_2 = -np.arccos((np.trace(R) - 1) / 2)
print(f"phi_2 = {phi_2:8.4f}\n")

angles = [np.degrees(np.arctan2(R_phi_eq[1, 0], R_phi_eq[0, 0])),
          np.degrees(np.arccos(R_phi_eq[0, 0])),
          np.degrees(np.arccos(R_phi_eq[1, 1])),
          -np.degrees(np.arccos(R_phi_eq[1, 1]))]
print("ans = " + '  '.join([f"{a:8.4f}" for a in angles]))