# -*- coding: utf-8 -*-
"""
Function which determines the stretch ratios and unit vectors for a
curvilinear coordinate system, per the discussion in the online supplement
to Chapter 2.
"""

import numpy as np


def curvi_vecs(r, phi, theta, delta_ang=1e-4, delta_r=1.0):
    """
    Determine stretch ratios and unit vectors for a curvilinear coordinate system
    (numerical Jacobian on spherical coordinates).

    Parameters
    ----------
    r, phi, theta : float
        Spherical coordinates (phi = polar angle, theta = azimuth).
    delta_ang : float
        Finite-difference step for angular variables (default 1e-4).
    delta_r : float
        Finite-difference step for r (default 1.0).

    Returns
    -------
    E : (3, 3) ndarray
        Unit-vector matrix (same as MATLAB: E = h \\ J).
    h : (3, 3) ndarray
        Diagonal matrix of stretch ratios.
    """
    def r_vec(rr, pp, tt):
        return np.array([
            rr * np.sin(pp) * np.cos(tt),
            rr * np.sin(pp) * np.sin(tt),
            rr * np.cos(pp),
        ], dtype=float)

    # Numerical evaluation of Jacobian on spherical coords (rows are derivatives)
    J = np.vstack([
        (r_vec(r + delta_r, phi, theta) - r_vec(r - delta_r, phi, theta)) / (2.0 * delta_r),
        (r_vec(r, phi + delta_ang, theta) - r_vec(r, phi - delta_ang, theta)) / (2.0 * delta_ang),
        (r_vec(r, phi, theta + delta_ang) - r_vec(r, phi, theta - delta_ang)) / (2.0 * delta_ang),
    ])

    h_sq = J @ J.T
    h_vals = np.sqrt(np.diag(h_sq))
    h = np.diag(h_vals)

    # MATLAB: E = h\J  (solve h * E = J)
    E = np.linalg.solve(h, J)
    return E, h
