# Trajectory_with_air_resistance.py
# Solves for the response of a projectile (golf ball) in free flight,
# subject to aerodynamic drag of the form f = c*v^2.
# The launch angle of the ball is varied and results are plotted for several trajectories.

import numpy as np
import matplotlib.pyplot as plt
from scipy.integrate import solve_ivp

def golf_eom(t, Z, k_m, g):
    """
    State Space equations of motion for golf ball in flight.
    Z_dot = golf_eom(t, Z, k_m, g)
    where k_m is the (c/m) constant and g is the gravitational constant.
    """
    N = 2
    q = Z[:N]
    q_dot = Z[N:]
    speed = np.linalg.norm(q_dot)
    # State Space EOM (Eq. (7))
    Z_dot = np.concatenate((q_dot, -k_m * speed * q_dot + np.array([0, -g])))
    return Z_dot

# Clear previous figures (MATLAB equivalent)
plt.close('all')

# Define system parameters
g = 9.807
v_term = 32  # Terminal Velocity in m/s
k_m = g / v_term**2

# Time setup
N_steps = 100
t_min = 0
t_max = 15
delta_t = (t_max - t_min) / N_steps
t_vals = np.linspace(t_min, t_max, N_steps)

# Initial velocity
v_0 = 80
psi_vals = np.radians([60, 45, 30, 15])  # Convert degrees to radians

# Set up plots
fig1, (ax1, ax2) = plt.subplots(2, 1, num=100)
fig2, ax3 = plt.subplots(num=110)

for psi in psi_vals:
    # Initial Position (x, z)
    q_0 = np.array([0, 0])
    # Initial Velocities
    q_dot_0 = v_0 * np.array([np.cos(psi), np.sin(psi)])
    # State Space Initial Condition
    Z_0 = np.concatenate((q_0, q_dot_0))

    # Solve using fixed time steps
    sol = solve_ivp(lambda t, Z: golf_eom(t, Z, k_m, g), [t_min, t_max], Z_0, t_eval=t_vals)

    q_vals = sol.y[:2, :].T
    q_dot_vals = sol.y[2:, :].T

    x_coord = q_vals[:, 0]
    z_coord = q_vals[:, 1]

    # Find instant of impact with the ground
    ind_impact = np.argmax(z_coord < 0) - 1 if np.any(z_coord < 0) else len(z_coord)

    # Plot x vs time
    ax1.plot(sol.t[:ind_impact], x_coord[:ind_impact], label=f"{np.degrees(psi):.0f}°")
    ax1.set_xlabel("Time (s)")
    ax1.set_ylabel("x (m)")
    ax1.set_ylim(0, 160)

    # Plot z vs time
    ax2.plot(sol.t[:ind_impact], z_coord[:ind_impact], label=f"{np.degrees(psi):.0f}°")
    ax2.set_xlabel("Time (s)")
    ax2.set_ylabel("z (m)")
    ax2.set_ylim(0, 150)

    # Plot trajectory (x vs z)
    ax3.plot(x_coord, z_coord, label=f"{np.degrees(psi):.0f}°")

# Final formatting for plots
ax1.legend()
ax2.legend()
ax3.set_xlabel("x (m)")
ax3.set_ylabel("z (m)")
ax3.set_xlim(0, 150)
ax3.set_ylim(0, 90)
ax3.set_aspect('equal')
ax3.legend()
plt.show()
