% This program is stored in file Squeeze_gear_response.m
% 
% Generalized coordinates are angle of rack C = theta, 
%      rotation of gear A = phi, and position of rack B = x_B, and the EOM
%      are defined in G_vec_squeeze.m 
% 
% System parameters and initial conditions
%      for the generalized coordinates must be entered below,
%      as well as the maximum time for a solution and the number
%      of time steps.
clear all
% Define system parameters here:
p.R = 0.1;
p.L = 0.4;
p.m_A = 2; 
kappa_A = 0.08;
p.I_A = p.m_A ^ kappa_A^2;
% p.m_B = 0.25; % Alternative value
p.m_B = 2.5;
p.m_C = 3;
kappa_C = 0.26;
p.I_C = p.m_C * kappa_C^2;
v_B_crit = 30;

% Uncomment this to loop over force values to find the desired value.
% Force_vals = [1000 : 1000 : 50000];
% Run this to see only the solution
Force_vals = 31089;
for n_force = 1 : length(Force_vals)
    p.Force = Force_vals(n_force);
    % Set initial conditions:
    t_0 = 0;
    q_0 = [pi/2  0  0]'; % Both q_0 and q_dot_0 should be column vectors.
    q_dot_0 = [0  0  0]';
    N = 3;
    % ode45 can decide the instants at which to output the response.
    %      Instead, the procedure implemented here uses a fixed time step.
    N_steps = 10000; % Number time steps for response output
    t_min = t_0;
    t_max = 0.1;
    delta_t = (t_max - t_min)/N_steps;
    t_vals = [0 : N_steps - 1] * delta_t;
    
    % The q and q_dot values at each instant are saved as columns
    %      in rectangular arrays.
    q_vals = zeros(N, N_steps); % Dedicate memory space
    q_dot_vals = zeros(N, N_steps);
    % Save starting values
    q_vals(:, 1) = q_0;
    q_dot_vals(:, 1) = q_dot_0;
    Z_prev = [q_0;  q_dot_0];
    Z_dot = G_vec_squeeze(Z_prev, t_vals(1), p);
    q_2dot_vals(:, 1) = Z_dot(N + 1 : 2 * N, 1);
    options = odeset('RelTol', 1.0e-6,'AbsTol',1.0e-6);
    % Begin time stepping
    for m = 2 : N_steps
        % Use default error tolerances. See Matlab documentation.
        [t_ode, Z_ode] = ode45(@(t, Z) G_vec_squeeze(Z, t, p), ...
            [t_vals(m - 1)  t_vals(m)], Z_prev, options);
        % An inconsistency of ode45 is that Z_prev should be a column vector,
        %      but the output array Z_ode consists of rows of Z
        %      corresponding to the instants contained in t_ode.
        % The response at time t(m) as the last row of Z_ode.
        num_evals = length(t_ode);
        % Z for next step is the same as Z at t(m).
        Z_prev = Z_ode(num_evals, :)'; % Convert from row to column storage.
        % Evaluate accelerations
        Z_dot = G_vec_squeeze(Z_prev, t_ode(num_evals), p);
        q_vals(:, m) = Z_prev(1 : N);
        q_dot_vals(:, m) = Z_prev(N + 1 : 2 * N)';
        q_2dot_vals(:, m) = Z_dot(N + 1 : 2 * N);
        theta_out = q_vals(1, m);
        x_A_out = p.R * cot(theta_out/2);
        x_B_out = q_vals(3, m);
%       Stop when right end of rack has passed the center of the gear
        if x_B_out > x_A_out
            m_max = m;
            t_vals = t_vals(1 : m_max);
            q_vals = q_vals(:, 1 : m_max);
            q_dot_vals = q_dot_vals(:, 1 : m_max);
            q_2dot_vals = q_2dot_vals(:, 1 : m_max);
%        Save v_b values at at the instant x_B > x_A
            v_B_last(n_force) = q_dot_vals(3, m_max);
            break, break
        end
    end
%     Check no slip at point H
    theta = q_vals(1, :);
    s = p.R * cot(theta/2);
    phi = q_vals(2, :);
    x_B = q_vals(3, :);
    d_theta = q_dot_vals(1, :);
    d_phi = q_dot_vals(2, :);
    d_x_B = q_dot_vals(3, :);
    v_H_g_x = d_x_B - p.R * d_phi .* (1 + cos(theta));
    v_H_g_y = - p.R * d_phi .* sin(theta);
    v_H_r_x = - s .* d_theta .* sin(theta);
    v_H_r_y = s .* d_theta .* cos(theta);
    figure(1000)
    subplot(2, 1, 1)
    plot(t_vals, v_H_g_x, '-r', t_vals, v_H_r_x, '--b')
    xlabel('Time (s)'); ylabel('X-Comp. of Velocity')
    subplot(2, 1, 2)
    plot(t_vals, v_H_g_y, '-r', t_vals, v_H_r_y, '--b')
    xlabel('Time (s)'); ylabel('Y-Comp. of Velocity')
end

% The calculation uses multiple force values, to graph v_B as a function of F.
% A single F value graphs the response at that F
if length(Force_vals) > 1
    figure(10)
    subplot(2, 1, 1)
    plot(Force_vals/1000, v_B_last, 'linewidth', 1)
    [min_diff, N_crit] = min(abs(v_B_last - v_B_crit));
    xlabel('Force F (kN)')
    ylabel('(v_B)_f (m/s)')
    grid on
    % Use linear interpolation to obtain an improved estimate
    slope = (v_B_last(N_crit + 1) - v_B_last(N_crit - 1))/...
        (Force_vals(N_crit + 1) - Force_vals(N_crit - 1));
    F_crit = Force_vals(N_crit - 1) + (v_B_crit - v_B_last(N_crit - 1))/slope;
    disp(['Critical force = ', num2str(F_crit)]);
    
else
    
    theta = q_vals(1, :);
    phi = q_vals(2, :);
    x_B = q_vals(3, :);
    x_A = p.R * cot(theta/2);
    x_B_dot = q_dot_vals(3, :);
    theta_dot = q_dot_vals(1, :);
    phi_dot = q_dot_vals(2, :);
    x_B_dot = q_dot_vals(3, :);
    theta_2dot = q_2dot_vals(1, :);
    phi_2dot = q_2dot_vals(2, :);
    x_B_2dot = q_2dot_vals(3, :);
    
    figure(100)
    subplot(3, 1, 1)
    yyaxis left
    plot(t_vals * 1000, x_B, '-r*', t_vals * 1000, x_A, '-bx', 'MarkerSize', 3, ...
        'MarkerIndices', 10 : 5 : size(t_vals), 'linewidth', 1)
    ylabel('x (meters)')
    yyaxis right
    plot(t_vals * 1000, theta * 180/pi, 'k--', t_vals * 1000, phi * 180/pi, 'g--o', ...
        'MarkerSize', 2, 'MarkerIndices', 10 : 10 : size(t_vals), 'linewidth', 1)
    ylabel('Angle (deg)')
    legend('x_B', 'x_A', '\theta', '\phi')
    
    subplot(3, 1, 2)
    yyaxis left
    plot(t_vals * 1000, x_B_dot, '-r', 'linewidth', 1)
    ylabel('dx_B/dt (m/s)')
    yyaxis right
    plot(t_vals * 1000, theta_dot, '--b', t_vals * 1000, phi_dot, '--g', 'linewidth', 1)
    ylabel('ang vel (rad/s)')
    legend('dx_B/dt', 'd\theta/dt', 'd\phi/dt')
    
    subplot(3, 1, 3)
    yyaxis left
    plot(t_vals * 1000, x_B_2dot, '-r', 'linewidth', 1)
    ylabel('d^2x_b/dt^2 (m/s^2)')
    yyaxis right
    plot(t_vals * 1000, theta_2dot, '--b', 'linewidth', 1)
    xlabel('time (ms)')
    ylabel('d^2\theta/dt^2 (rad/s^2)')
    legend('d^2x_B/dt^2', 'd^2\theta/dt^2')
    
end
