function [q,  q_dot, t_max, out_file] = Rolling_disk_ic(ic_case, Radius, kappa) 
% 
% Set the initial generalized coordinates and generalized velocities for a 
% rolling disk.
%
%  q = [X  Y  psi  theta  phi]'
%  q_dot = [X_dot  Y_dot  psi_dot  theta_dot  phi_do]'
%
% Set specific case by altering any values, 
%   except that X_dot and Y_dot are obtained from the constraint equations

if ic_case == 1
    %---------------  First initial condition case  ------------------------
    % Initial conditions match the steady precession solution.
    out_file = 'steady.dat';
%     Set initial q
    rho = 5 * Radius;
    psi = 0;
    phi = 0;
    theta = pi/3;  % Set nominal value of nutation angle
    X = 0;
    Y = rho;
 % Set angular velocities   
    psi_dot = sqrt(2 * 9.807 * Radius^2 * cot(theta)/(2 * rho * (Radius^2 + kappa^2)...
        + kappa^2 * Radius * cos(theta)));
    phi_dot = -(rho/Radius + cos(theta)) * psi_dot
    theta_dot = 0
    % Set maximum time for integration:
    t_max = 10;

elseif ic_case == 2
    %---------------  Second initial condition case  ------------------------
    % Disturb the steady precession solution by setting all variables other
    % than theta and theta_dot to steady precession values for some other theta.
    out_file = 'quasi_steady.dat';
    rho = 5 * Radius;
    psi = 0;
    phi = 0;
    theta = pi/3;  % Set nominal value of nutation angle
    X = -rho*sin(psi);
    Y = rho*cos(psi);
 % Set angular velocities   
    psi_dot = sqrt(2 * 9.807 * Radius^2 * cot(theta)/(2 * rho * (Radius^2 + kappa^2)...
        + kappa^2 * Radius * cos(theta)));
    phi_dot = -(rho/Radius + cos(theta)) * psi_dot
    theta_dot = - 0.5 * psi_dot;  % Nonzero nutational velocity
    
    theta = pi/6;  % Change theta to non-steady value
    
    % Set maximum time for integration:
    t_max = 10; % Augmented cpu time = 131.6
elseif ic_case == 3
    %---------------  Third initial condition case  ------------------------
    % Examine stability of planar rolling.
    % Set initial psi_dot=0, phi_dot very low, theta slightly smaller
    % than pi/s, and theta_dot equals zero.
    out_file = 'stability_orthogonal.dat';
% Set initial q
    X = 0; 
    Y = 0;
    psi = 0;
    phi = 0;
    theta = pi/2 - 0.01;
 % Set angular velocities   
    v_cr = sqrt(9.807 * Radius/3);
    v = 0.2 * v_cr
    psi_dot = 0;
    phi_dot = -0.2 * sqrt(9.807/(3 * Radius));
    theta_dot = 0;
    % Set maximum time for integration:
%     t_max = 1000;
    t_max = 50;
    %
    %-----------------  End definition of initial conditions  ---------------------
end
% Define initial generalized coordinate vector
q = [X  Y  psi  theta  phi ]';

% Evaluate Jacobian constraint matrix at initial time
% Note that a_dot is also computed by the a_deriv function, but that 
% value is not needed to set initial conditions, so q_dot is set to zero
[a, a_dot] = a_a_dot_disk([q; zeros(5,1)], Radius);

% Solve for initial values of (d/dt)X and (d/dt)Y
a_XY = a(:, 1 : 2);
a_ang = a(:, 3 : 5);
XY_dot = -a_XY\(a_ang * [psi_dot  theta_dot  phi_dot]');

% Form initial value of q_dot
q_dot = [XY_dot; psi_dot;  theta_dot;  phi_dot];
end

