% Solution of differential equations for a wobbling disk 
% Radius is the radius of the disk, kappa is the radius of gyration
% Units are SI -- Need to change g value in rolling_disk_f_mat.m function 
% file to use U.S. Customary units.
%
% {Z} is current state-space vector: 
%     number of generalized coordinates is 5, so size of {Z} is 10
% Sequence of generalized coordinates is:
%     Z(1) = X, Z(2) = Y, Z(3) = psi, Z(4) = theta, x(5) = phi
% Sequence of generalized velocities is:
%     Z(6) = d(X), Z(7) = d(Y), Z(8) = d(psi), Z(9) = d(theta), Z(10) = d(phi)
% Use m-file: resp_plt(t_val, x_val) to save and plot data

clear all

Radius = 0.25;
kappa = Radius/sqrt(2);

% Set up initial conditions.
ic_case = 0;
while (ic_case < 1) | (ic_case >3)
    disp('Enter the initial condition case')
    disp('  1 ==> Steady precession')
    disp('  2 ==> Disturbance of steady precession')
    disp('  3 ==> Stability of planar rolling')
    ic_case = input('Enter 1, 2, or 3 __ ');
end
[q,  q_dot, t_max, out_file] = Rolling_disk_ic(ic_case, Radius, kappa);
Z_0 = [q; q_dot];

disp(' ')
disp(['Max run time is ', num2str(t_max)])

% Set time increment
dt = 0.005;
i_max = floor(t_max/dt) + 1;

% Set tolerance for ODE routine
% options = odeset('RelTol', 1.0e-12,'AbsTol',1.0e-12); 
options = odeset('RelTol', 1.0e-8, 'AbsTol', 1.0e-8); 

% Track CPU time
t0_cpu = cputime; 

%-----------------------------------------------------------------------------------------
% Integration loop
for i = 1:i_max
	t_val(i, 1) = (i - 1) * dt;
	Z_val(i, :) = Z_0';

    % Check conservation of energy:
	Tot_en(i) = 0.5 * (Z_0(6)^2 + Z_0(7)^2 + (0.5 * kappa^2 ... 
			+ Radius^2 * cos(Z_0(4))^2) * Z_0(9)^2  ...
			+ 0.5 * kappa^2 * Z_0(8)^2 * sin(Z_0(4))^2 ...
			+ kappa^2 * (Z_0(8) * cos(Z_0(4)) + Z_0(10))^2) ...
			+ 9.807 * Radius * sin(Z_0(4));
% Check velocity constraint error
    [a, a_dot] = a_a_dot_disk([Z_0], Radius);
    constraint_err(i)= norm(a * Z_0(6:10)/5);

% Test if disk has fallen on its side
	theta = Z_0(4);
	if (theta >= 0)*(theta <= pi) == 0, break, end

% Set start and end times for calling o.d.e. solver
	t0 = (i - 1) * dt;
	tf = i * dt;
%    if mod(i, 100) == 0
%        disp(num2str([t0  Z_val(i, 1)  Z_val(i, 2)  theta  Tot_en(i)], '%15.6f'))
%    end
% Call Runge-Kutta ODE solver, then plot result
	[t, Z] = ode15s(@(t, Z) G_vec_disk_aug(t, Z, Radius, kappa), [t0  tf], Z_0, options);    
% Save current t and {q}
	n_steps = length(t);
	Z_0 = Z(n_steps, :)';
end

Rolling_disk_plot(t_val, Z_val);

t_cpu = cputime - t0_cpu;
disp([''])
disp(['CPU time for this run = ', num2str(t_cpu)])

disp(' ')
disp(['Errors: max(E - E_0)/E_0 = ', num2str(max(Tot_en-Tot_en(1))/Tot_en(1)), ...
    ',  max(norm([a] * dq/dt)) = ', num2str(max(constraint_err))]) 