% This program is stored in file Friction_linkage_response.m
% It evaluates the response of a linkage whose arm AB rotates about end A,
%   and whose end B is pinned to bar BC that  slides through collar D, which
%   is stationary. Frictional resistance is developed at the collar.
%   A torque causes bar AB to rotate. 
%   Generalized coordinates are theta for bar AB and phi for bar BC.
%   The state-space vector is Z = [theta, phi, theta_dot, phi_dot]'.
clear all

% Define system parameters here: These ones are varied in the discussion in
% the book:
% mu_k = 0;
mu_k = 0.5; % Friction Coefficient
% Applied Torque Gamma
% Gam = 1000; t_max=2; 
% Gam = 3000; t_max=0.5; 
% Gam = 1741.2991; t_max=2; % Use with mu_k=0
% Gam = 1741.2992; t_max=2;
% Gam = 2165.7; t_max=2; % Use with mu_k = 0.5
Gam = 2165.72; t_max=2; % Use with mu_k=0.5
% Largest torque for return to theta = 0 is Gam = 1741.299 N-m
% Gam = input('Enter the amplitude of the torque:  ');

% Remaining parameters
L = 0.5; % m
L_0 = 3 * L; % m, Unstretched length of the spring
beta_L = L_0/L;
m_AB = 6; % kg
m_BC = 24; % kg
k_stiff = 10e+3; % N/m

% Set initial conditions:
t_0 = 0;
theta_0 = 0 * pi/180;
alpha_0 = sqrt(5 - 4 * cos(theta_0));
phi_0 = asin(sin(theta_0)/alpha_0);
q_0 = [theta_0;  phi_0]; % Both q_0 and q_dot_0 should be column vectors.
% q_0 = [0;  0];
% q_0 = [2.580818364924015;  0.184689179424080]; % Static equilibrium value at max Gam
q_dot_0 = [0;  0];

N = 2;
% Selection of time step.
% ode45 can decide the instants at which to output the response.
%      The procedure implemented here reports the response at a fixed time
%      step, but allows ode45 to select any intermediate timesteps, which
%      are not recorded.
Omega = sqrt(k_stiff/(m_AB + m_BC));
T_nat = sqrt(2 * pi/Omega);
delta_t = T_nat/1000;
% t_max = 3.2 * T_period;
% t_max = 0.9001;
% t_max = 1.5;
% t_max = input('Enter the maximum execution time:  ');
max_steps = ceil(t_max/delta_t); % Number time steps for response output
t_vals = [0 : max_steps] * delta_t;
% The q and q_dot values at each instant are saved as columns
%      in rectangular arrays.
q_vals = zeros(N, max_steps); % Dedicate memory space
q_dot_vals = zeros(N, max_steps);
q_2dot_vals = zeros(N, max_steps);
% Save starting values
q_vals(:, 1) = q_0;
q_dot_vals(:, 1) = q_dot_0;
Z_0 = [q_0;  q_dot_0];
Z_prev = Z_0;
N_D = 0;
N_D_vals(1) = N_D;
% Evaluate initial acceleration in order to have continuous functions at t = 0.
[Z_dot] = G_vec_linkage_friction(Z_0, 0, N_D, m_AB, m_BC, L, beta_L, k_stiff, Gam, mu_k);
q_2dot_vals(:, 1) = Z_dot(N + 1 : 2 * N, 1);   
options = odeset('RelTol', 1.0e-8,'AbsTol',1.0e-8); 
% Begin time stepping
for m = 2 : max_steps + 1
    t_0 = t_vals(m - 1);
    t_f = t_vals(m);
    [t_ode, Z_ode] = ode45(@(t, Z) G_vec_linkage_friction(Z, t, N_D, m_AB, m_BC, L, ...
        beta_L, k_stiff, Gam, mu_k), [t_0, t_f], Z_prev, options);
    % The response at time t_f is the last row of Z_ode.
    num_evals = length(t_ode);
%     disp(['m=',  num2str(m),  '  t=', num2str(t_ode(num_evals)), '  Z=', ...
%         num2str(Z_ode(num_evals, :)), '   N_D=', num2str(N_D)])
    % Initial value of Z for next step is the same as Z at t_f.
    Z_prev = Z_ode(num_evals, :)'; % Convert from row to column storage.
    N_D_prev = N_D;
    [N_D, q_2dot] = Friction_linkage_Eval_N_d(Z_prev, t_f, N_D_prev, m_AB, m_BC, L, beta_L, k_stiff, Gam, mu_k);
    q_vals(:, m) = Z_prev(1 : N);
%     disp(['After Eval_N_D:  N_D=', num2str(N_D), '   Z=', num2str(Z_ode(num_evals, :))]);
    q_dot_vals(:, m) = Z_prev(N + 1 : 2 * N);
    % Save accelerations and normal force. This data is not used in the example.
    q_2dot_vals(:, m) = q_2dot;
    N_D_vals(m) = N_D;
    if mod(m, 100) == 0
        disp(['m = ', num2str(m)])
    end
end

m_max = length(t_vals);
% mm = [0 : m_max - 1];
theta = q_vals(1, :);
phi = q_vals(2, :);
theta_dot = q_dot_vals(1, :);
phi_dot = q_dot_vals(2, :);
theta_2dot = q_2dot_vals(1, :);
phi_2dot = q_2dot_vals(2, :);
F_spr = k_stiff * L * (2 - sqrt(5 - 4 * cos(theta)));
time = t_vals';
err_config = sin(theta + phi) - 2 * sin(phi);
err_vel = cos(theta + phi) .* sin(phi) .* theta_dot - sin(theta) .* phi_dot;
disp(' ')
disp(['Max configuration constraint error = ', num2str(max(abs(err_config)))])
min_phi = min(abs(phi(200 : m_max))) * 180/pi;
min_theta = min(abs(theta(200 : m_max))) * 180/pi;
max_phi = max(abs(phi(200 : m_max))) * 180/pi;
max_theta = max(abs(theta(200 : m_max))) * 180/pi;
disp(' ')
disp(['min theta = ', num2str(min_theta), ', min phi = ', num2str(min_phi)]) 
disp(['max theta = ', num2str(max_theta), ', max phi = ', num2str(max_phi)]) 
disp(' ')

figure(1)
% Wrap data into 360 deg range
subplot(3, 1, 1)
plot(time * 1000, mod(theta * 180/pi + 180, 360) - 180, 'r-', time * 1000, phi * 180/pi, 'b-o', ...
    'MarkerSize', 2, 'MarkerIndices', [100 : 100 : m_max], 'linewidth', 0.75)
xlim([0  max(time) * 1000])
if max(theta) > pi * 185/180
    ylim([-190 190])
else
    ylim([-35 190])
end
yticks([-180 : 30: 180])
ylabel('Angle \theta (deg)')
legend('\theta', '\phi')
title(['\Gamma=',num2str(Gam, 11),', k=',num2str(k_stiff),', \mu=',num2str(mu_k), ...
    ', max(\phi)=', num2str(max_phi)])

subplot(3, 1, 2)
plot(time * 1000, theta_dot, '-r', time * 1000, phi_dot, 'b--o', ...
    'MarkerSize', 2, 'MarkerIndices', [100 : 100 : m_max], 'linewidth', 1)
xlim([0  max(time) * 1000])
xlabel('Time (ms)')
ylabel('Ang. Vel. (rad/s)')
legend('d\theta/dt', 'd\phi/dt')
subplot(3, 1, 3)
plot(time * 1000, N_D_vals, '-r', time * 1000, F_spr, 'b--o', 'MarkerSize', 2, 'MarkerIndices', ...
    [100 : 100 : m_max], 'linewidth', 0.75)
ylabel('Forces (N)')
xlim([0  max(time) * 1000])
%     ylim([-6001  6001])
legend('N_D', 'k\Delta')
name_fig = ['Friction Gam=', num2str(Gam, 12), '.pdf'];
saveas(gcf, name_fig, 'pdf')

% figure(2)
% subplot(3, 1, 1)
% Plot theta in full range without wrapping
% plot(time * 1000, theta * 180/pi, 'r-', time * 1000, phi * 180/pi, 'b-o', ...
%     'MarkerSize', 2, 'MarkerIndices', [100 : 100 : m_max], 'linewidth', 0.75)
% xlim([0  max(time) * 1000])
% % ylim([0  100])
% ylabel('Angle (deg)')
% legend('\theta', '\phi')
% title(['\Gamma=',num2str(Gam, 18), ', k=',num2str(k_stiff),', \mu=',num2str(mu_k), ...
%     ', max(\phi)=', num2str(max_phi)])
% 
% subplot(3, 1, 2)
% plot(time * 1000, theta_dot, '-r', time * 1000, phi_dot, 'b--o', ...
%     'MarkerSize', 2, 'MarkerIndices', [100 : 100 : m_max], 'linewidth', 0.75)
% xlim([0  max(time) * 1000])
% ylabel('ang vel (rad/s)')
% legend('d\theta/dt', 'd\phi/dt')

% subplot(3, 1, 2)
% plot(t_vals * 1000, theta_2dot, '--b', 'linewidth', 0.75)
% xlabel('time (ms)')
% ylabel('d^2\theta/dt^2 (rad/s^2)')
% legend('d^2x_B/dt^2', 'd^2\theta/dt^2')

%%%%%%%%%%%%%%%%
% Evaluate conditions for max theta and sticking when mu=0
thetas = [0 : 0.2 : 180] * pi/180;
alphas = sqrt(5 - 4 * cos(thetas));
deltas = (1 - alphas) * L;
V = 0.5 * k_stiff * deltas.^2;
Gamma_static = (2./alphas) .* k_stiff * L .* abs(deltas) .* sin(thetas);
% Plot the result
figure(11)
plot(thetas * 180/pi, V./thetas, 'r', thetas * 180/pi, Gamma_static, 'b--')
xlabel('\bf\Theta (deg)'); ylabel('\bfTorque \Gamma (N/m)')

% Solve
alpha_func = @(theta) sqrt(5 - 4 * cos(theta));
delta_func = @(theta) (1 - alpha_func(theta)) * L;
V_func = @(theta) 0.5 * k_stiff * delta_func(theta).^2;
Gam_stat_func = @(theta) (2./alpha_func(theta)) .* k_stiff * ...
    L .* abs(delta_func(theta)) .* sin(theta);
F_lock = @(theta) Gam_stat_func(theta) - V_func(theta)./theta;
theta_start = 140 * pi/180;
th_lock = fzero(F_lock,theta_start)
Gam_lock = Gam_stat_func(th_lock)
