% Response_rack_gear.m 
% This script computes the response of a holonomic system that is described
% by unconstrained generalized coordinates. 
% One must first create a function G_vec that returns d{Z}/dt for given
% values of the states Z and time t.  Then, any system parameters are
% passed to G_vec and the initial conditions for the generalized
% coordinates are entered below as well as the initial and final time for
% the solution. 
clear all
close all
% Define system parameters here:
p.m_r = 20; p.m_A=10; p.m_s=2; % masses in (kg)
p.kappa = 0.150; p.radius=0.200; p.e_offset=0.175; % lengths in (meters)
p.k_stiff = 20; % (N/m)
% p.k_stiff = 400;
p.freq = 4;% Forcing Frequency (rad/s)
p.F_mag = 25; % Forcing Amplitude (N)
Ig = p.m_A * p.kappa^2;
% Set initial conditions:
t_0 = 0;
q_0 = [2 * p.radius; 0]; % Both q_0 and q_dot_0 should be row vectors.
q_dot_0 = [0; 0];
N = length(q_0);
T_force = 2 * pi/p.freq;
% Call ode45 to compute the response Z=[q; q_dot] from t_0 to t_max
%   Use @(t, Z) to create a function of t and Z only for ODE45, 
%   while simultaineously passing the system parameters 
%   to G_vec_rack(t, Z, system parameters)
T_max = 16 * T_force; % time at which to end the integration
N_intervals = 16; %Number of intervls in a period
delta_t = T_force/N_intervals;
t_out = [0 : delta_t : T_max];
options = odeset('AbsTol', 1e-10, 'RelTol',1e-10);
[t_ode, Z_ode] = ode45(@(t, Z) G_vec_rack(Z, t, p),...
    t_out, [q_0; q_dot_0], options);
% [t_ode, Z_ode] = ode45(@(t, Z) G_vec_rack(Z, t, param),...
%     [0, T_max], [q_0; q_dot_0]);
% Create a plot of the response
x = Z_ode(:, 1);
theta = Z_ode(:, 2);
x_rack = x + p.e_offset * sin(theta) + p.radius * theta;
y_B = p.e_offset * cos(theta);
t_nondim = t_ode * p.freq/(2 * pi);
figure(10);
% subplot(3, 1, 1)
% plot(t_nondim, x, 'r',t_nondim, y_B, 'b:', ...
%     'linewidth', 0.75)
% xlim([0  T_max/T_force])
% ylim([-0.6 0.8])
% xticks([0 : 2 : T_max/T_force])
% yticks([-0.4 : 0.2 : 0.4])
% ylabel('Position (m)');
% grid
% legend('x_B', 'y_B'); 

subplot(2,1,1)
yyaxis left
plot(t_nondim, x, 'b --', t_nondim, x_rack, 'r-', 'linewidth', 0.75)
ylim([-0.8  1.6])
yticks([-0.8 : 0.8 : 1.6])
ylabel('Position (m)');
yyaxis right
plot(t_nondim, theta*180/pi, 'k:', 'linewidth', 1);
xlim([0  T_max/T_force])
ylim([-180 360])
xticks([0 : 2 : T_max/T_force])
yticks([-180 : 180 : 360])
% text(2, -0.25, '\theta')
xlabel('t/T_p_e_r_i_o_d'); 
ylabel('angle (deg)');
legend('x', 'x_r', '\theta')
grid

% Note - this is also interesting with a small offset, so that the
% oscillations are stronger.

%% Create an animation of the gear-rack system
% Define coordinates of parts when x = theta = 0 
% Origin of xyz is at the ground under the right end of the guide bar 

% Gear 
ths = [0 : 1 : 360] * pi/180; % Angles to points around gear circumference
xg_pts = p.radius * sin(ths);
yg_pts = p.radius * (1 + cos(ths));
% (xs_pts,ys_pts) are coordinates of corners of the rack at x = 0
xs_pts = ([-0.3, 0.3, 0.3, 0.1, 0.1, -0.1,...
    -0.1,-0.3,-0.3]) * p.radius;
ys_pts = ([0.1, 0.1, -0.1, -0.1, -2.2, -2.2,...
    -0.1, -0.1, 0.1] + 2.5) * p.radius;
% (xr, yr) are coordinates of corners of the rack 
%  with forward end at tangency with gear
xr_pts=[0, -4, -4, 0, 0]; % 4 meters long,
yr_pts=[0, 0, -0.1, -0.1, 0];
% Force Vector
Force = p.F_mag * sin(p.freq * t_ode);
% Evaluate position of the system at various instants
N_col = 2;
N_row = 3;
N_plot = 0;
fignum = 20;
% k_index = [1 : 30 : length(t_ode)];
% k_index = [1: 1 : 17];
k_index = [1 : 4 : 65];
for kk = 1 : length(k_index)
    k = k_index(kk);
    % Shift all point based in x and theta at t_ode(k)
    % Point B
    xB = x(k);
    % yB is vertical distance to pin B from the top of the rack
    yB = p.radius + p.e_offset * cos(theta(k));
    % Position of points on the circumference of the gear
    xg_center = x(k) + p.e_offset * sin(theta(k));
    xg = xg_center + xg_pts;
    yg = yg_pts;
    % T-Slider Geometry
    xs = xs_pts + x(k); % x-position.
    ys = ys_pts;
    % Rack Geometry
    % xr_pts are relative to the leading edge.
    xr = xr_pts + x_rack(k); 
    yr = yr_pts;
    
% Figures are N_row x N_col grid
% Start a new grid after grid is full
    if N_plot == N_row * N_col 
        fignum = fignum + 10;
        N_plot = 0;
    end
    N_plot = N_plot + 1;
    if N_plot == 1
        figure(fignum)
    end
    subplot(N_row, N_col, N_plot)
    %  Patch plots one or more filled 2D polygonal regions
    patch(xs, ys, 'b');
    patch(xr, yr, 'g');
    patch(xg, yg, 'r','FaceAlpha',0.5);
    % Single point for the pin
    line(xB, yB, 'Marker','o', 'MarkerFaceColor','y', 'MarkerSize',2)
    hold on;
    set(gca, 'XDir','reverse')
    xlim([-2,  3])
    %     ylim([0, 0.5])
    hold on
    plot([x(1), x(1)], [0, 3.5] * p.radius, 'k--', 'linewidth', 1)
    text(-0.75, 0.25, num2str(t_ode(k)/T_force))
    axis equal
end

% Evaluate acceleratioons and forces
for n = 1 : length(t_ode)
    Z_dot = G_vec_rack(Z_ode(n,:)', t_ode(n), p);
    x_dot(n) = Z_dot(1);
    theta_dot(n) = Z_dot(2);
    x_2dot(n) = Z_dot(3);
    theta_2dot(n) = Z_dot(4);
    x_A_2dot(n) = x_2dot(n) + p.e_offset * theta_2dot(n) * cos(theta(n))...
        - p.e_offset * theta_dot(n).^2 * sin(theta(n));
    x_rack_2dot(n) = x_A_2dot(n) + p.radius * theta_2dot(n); 
    % I*alpha for the gear, positive clockwise
    Moment_gear(n) = Ig * theta_2dot(n);
    % m * a for the slider
    Res_slider(n) = p.m_s * x_2dot(n);
    % m * a for the gear
    Res_gear(n) = p.m_A * x_A_2dot(n);
    % m * a for the rack
    Res_rack(n) = p.m_r * x_rack_2dot(n);
    Spring_force(n) = -p.k_stiff * (x(n)- 2 * p.radius);
    % Positive pin force is in the +x direction on the slider, 
    % so it is in the -x direction on te gear
    Pin_force(n) = Res_slider(n) - Spring_force(n);
    Force_ext(n) = p.F_mag * sin(p.freq * t_ode(n));
    % Positive rack force is in the +x direction on the gear,
    % so it is in the -x direction on the rack
    Rack_force(n) = -Res_rack(n) + Force_ext(n);
    % Moments are positive if clockwise
    Moment_pin(n) = Pin_force(n) * p.e_offset * cos(theta(n));
    Moment_rack(n) = Rack_force(n) * p.radius;
end
    
figure(30)
subplot(2,1,1)
plot(t_nondim, -Rack_force, '--r', t_nondim, Force_ext, ':b', ...
    t_nondim, Res_rack, '-k','linewidth', 1);
xlim([0  4])
xticks([0 : 4])
ylim([-30  40])
yticks([-30 : 20 : 40])
xlabel('t/T_F'); 
ylabel('Forces (N)');
legend('-F_r', 'Excitation', 'm_r*d^2x_r/dt^2')
grid

subplot(2,1,2)
plot(t_nondim, Pin_force, ':r', t_nondim, Spring_force, '--b',...
    t_nondim, Res_slider, '-k','linewidth', 1);
xlim([0, 4])
xticks([0 : 4])
% ylim([-2 * p.k_stiff, 2 * p.k_stiff])
ylim([-7  9.5])
yticks([-4 : 2 : 4])
xlabel('t/T_F'); 
ylabel('Forces (N)');
legend('F_B', '-k*x', 'm_s*d^2x/dt^2')
grid

figure(40)
subplot(2,1,1)
plot(t_nondim, Rack_force, ':r', t_nondim, -Pin_force, '--b',...
    t_nondim, Res_gear, '-k','linewidth', 1);
xlim([0, 8])
xticks([0 : 8])
% ylim([-2 * p.k_stiff, 2 * p.k_stiff])
ylim([-8  8])
yticks([-4 : 4 : 4])
xlabel('t/T_F'); 
ylabel('Forces (N)');
legend('F_B', '-k*x', 'm_s*d^2x/dt^2')
grid

subplot(2,1,2)
plot(t_nondim, Moment_pin, '--r', t_nondim, Moment_rack, ':b', ...
    t_nondim, Moment_gear, '-k','linewidth', 1);
xlim([0  4])
xticks([0 : 4])
ylim([-2  2]) 
yticks([-1 : 1 : 1])
xlabel('t/T_p_e_r_i_o_d'); 
ylabel('Forces (N)');
legend('M_B', 'M_r', 'I_A*d^2\theta/dt^2')
grid

