% Balanced_gyro_responses.m
%
% This program is based on the program in Response_Unconstrained.m
%      It evaluates the response of a gyroscopic system subjectd 
%      to precessional torque.
%
% System parameters and initial conditions
%      for the generalized coordinates are entered below,
%      as well as the time at which to end the solution.
%
% The value of d{Z}/dt is computed in G_vec_gyro, 
%      which calls M_mat_gyro and F_vec_gyro

clear all
close all
% Use Times New Roman for all text
set(groot, 'DefaultAxesFontName', 'Times New Roman')
set(groot, 'DefaultTextFontName', 'Times New Roman')
set(groot, 'DefaultLegendFontName', 'Times New Roman')
% Use LaTeX for all text interpreters
set(groot, 'DefaultTextInterpreter', 'latex')
set(groot, 'DefaultAxesTickLabelInterpreter', 'latex')
set(groot, 'DefaultLegendInterpreter', 'latex')


% Define system parameters here:
p.mass = 0.16;  p.I_1 = 5e-5;  p.I_2 = 2.6e-5; 
p.L = 0.0;  
p.phi_dot = 1e4 * 2 * pi/60;
p.I_2_eff = p.I_2 + p.mass * p.L^2;  
p.mgL = p.mass * 9.807 * p.L;
p.Gam_mag = 2; 
p.freq = 20;

% Set initial conditions:
t_0 = 0; 
psi_0 = 0; theta_0 = pi/3;
psi_dot_0 = 0; theta_dot_0 = -0.01;

% Intialize ode solver
Z_0 = [psi_0; theta_0; psi_dot_0; theta_dot_0];
t_min = t_0;
T_max = 2;

% Set error tolerances for ode solver
% opts = odeset('AbsTol',1e-6,'RelTol',1e-3);
opts = odeset('AbsTol',1e-12,'RelTol',1e-12);

% Use ode15s or ode45 by editing below. 
[t_ode, Z_ode] = ode15s(@(t, Z) G_vec_gyro(Z, t, p), ...
    [t_0, T_max], Z_0, opts);
% A row of Z_ode is {Z} at an instant.
% A column of Z_ode is an element of {Z} at all instants
psi = Z_ode(:, 1);
theta = Z_ode(:, 2);
psi_dot = Z_ode(:, 3);
theta_dot = Z_ode(:, 4);
% Evaluate angular momentum relative to xyz
H_O = [p.I_2 * psi_dot .* sin(theta), p.I_1 * (psi_dot .* cos(theta) - p.phi_dot)];
% Transform H_O to x'y'z'
H_O_xyz_prime(:, 1) = cos(theta) .* H_O(:, 1) - sin(theta) .* H_O(:, 2);
H_O_xyz_prime(:, 2) = sin(theta) .* H_O(:, 1) + cos(theta) .* H_O(:, 2);
% Evaluate stiffnes parameter divided by I_2
Kappa_I_2 = (p.I_1/p.I_2 - 1) * cos(2 * theta) .* psi.^2 ...
    - (p.I_1/p.I_2) * cos(theta) .* p.phi_dot .* psi_dot;

figure(1)
subplot(3, 1, 1)% 
plot(t_ode, theta * 180/pi, 'r-', 'linewidth', 0.5)
ylabel('$\theta$ (degrees)')
grid on
ylim([0  360])
yticks([0 : 180 : 360])
subplot(3, 1, 2)
plot(t_ode, psi_dot, 'k:', t_ode, psi_dot .* cos(theta) - p.phi_dot, 'r-', ...
     t_ode, psi_dot .* sin(theta), 'b--', 'linewidth', 0.5)
ylabel('Rotation rate (rad/s)')
ylim([-4000, 4000])
yticks([-4000 : 4000 : 4000])
grid on
legend('$d\psi/dt$', '$\omega_z$', '$\omega_x$')
subplot(3, 1, 3)
plot(t_ode, H_O(:, 1), 'b-', t_ode, H_O(:, 2), 'r--', 'linewidth', 0.5)
ylim([-0.2, 0.05])
yticks([-0.2 : 0.1 : 0.1])
ylabel('$H_O$ (kg-m$^2$/s) in xyz')
xlabel('Time (s)')
legend('$(\bar{H}_O) \cdot \bar{i}$', '$(\bar{H}_O) \cdot \bar{k}$')
grid on
% title('xyz components of H_O')

figure(10)
subplot(3, 1, 1) 
plot(t_ode, theta * 180/pi, 'r-', 'linewidth', 0.5)
ylabel('$\theta$ (degrees)')
grid on
ylim([0  360])
yticks([0 : 180 : 360])
subplot(3, 1, 2)
plot(t_ode, psi_dot, 'r--', 'linewidth', 0.5)
ylim([-3000, 3000])
yticks([-2000 : 2000 : 2000])
ylabel('$\dot{\psi}$ (rad/s)')
% xlabel('Time (s)')
grid on
subplot(3, 1, 3)
plot(t_ode, H_O_xyz_prime(:, 1), 'b--', t_ode, H_O_xyz_prime(:, 2), 'r-', 'linewidth', 0.5)
ylim([-0.1, 0.2])
yticks([-0.1 : 0.1 : 0.2])
ylabel('$(H_O)$ (kg-m$^2$/s)') % '$(H_O)$ (kg-m$^2$/s) in $x^{\prime}y^{\prime}z^{\prime}$'
xlabel('Time (s)')
legend('$(\bar{H}_O) \cdot \bar{i}^{\prime}$', '$(\bar{H}_O) \cdot \bar{k}^{\prime}$')
grid on
% title('x''y''z'' components of H_O')

figure(20)
subplot(3, 1, 1)
plot(t_ode, theta * 180/pi, 'r-', 'linewidth', 0.5)
ylabel('$\theta$ (degrees)', 'Color','black')
grid on
ylim([0, 360])
yticks([0 : 180 : 360])
subplot(3, 1, 2)
plot(t_ode, Kappa_I_2, 'r')
ylabel('{\it k}/I_2 (rad/s)','Interpreter','tex')
ylim([-4e6, 8e6])
yticks(-4e6 : 4e6 : 8e6)
xlabel('Time (s)')

% Draw angular momentum and coordinate axes
figure(100); clf(100)
hold on
times=[0,0.32, 0.64, 0.96, 1.26, 1.58]; % include 6 times
for nplot = 1 : 6
    subplot(2, 3, nplot)
%     time = input('Enter the time:  ')
    [min_time, I_time] = min((abs(t_ode - times(nplot))));
    theta_t = theta(I_time);
    disp(['I=', num2str(I_time), ', theta=', num2str(theta_t * 180/pi),...
        ', t =', num2str(t_ode(I_time))])
    H_O_t = H_O_xyz_prime(I_time, :);
    e_H = H_O_t/norm(H_O_t);
    e_X = [1, 0];
    e_Z = [0, 1];
    e_x = [cos(theta_t), sin(theta_t)];
    e_z= [-sin(theta_t), cos(theta_t)];
%  Reverse X direction due to looking down the Y axis   
    hold on
    quiver(0, 0, -e_X(1), e_X(2), 'k'); text(-e_X(1)*1.05, e_X(2)*1.05,'X')
    quiver(0, 0, -e_Z(1), e_Z(2), 'k'); text(-e_Z(1)*1.05, e_Z(2)*1.05,'Z')
    quiver(0, 0, -e_x(1), e_x(2), 'b'); text(-e_x(1)*1.05, e_x(2)*1.05,'x')
    quiver(0, 0, -e_z(1), e_z(2), 'b'); text(-e_z(1)*1.05, e_z(2)*1.05,'z')
    H_dir = quiver(0, 0, -H_O_t(1), H_O_t(2), 10); % last arguemnt is scale factor
    H_dir.Color = 'red';
    title(['t=',num2str(t_ode(I_time),2), ', $\dot{\psi}$=', num2str(psi_dot(I_time),4)])
    xlim([-1.2, 1.2])
    ylim([-1.2, 1.2])
    axis square
end
hold off


