% Script file Falcon_rocket.m
% Evaluate the motion of the Space-X Falcon rocket
% Vertical path. Include variable mass, variable density, and variable gravity.
% Then remove each refinement.
clear all
%%%%%%%%%%%%%%%%
% Parameters
%%%%%%%%%%%%%%%%
m_total = 549e3; % kg
m_prop = 385e3; % kg
t_burn = 185; % seconds
m_dot = m_prop/t_burn;
Thrust = 6e6; % N
x_area = 40; % m^2, 5.04 m diameter fairing
C_d = 0.4; % drag coefficient
rho_0 = 1.29; % kg/m^3
H = 10400;  % Density of air is rho_0 * exp(-h/H);
R_e = 6378e3;

%%%%%%%%%%%%%%%%
% Set initial conditions:
t_0 = 0;
y_0 = 0;
v_0 = 0;
% Set time steps for ode45
N_steps = t_burn; % One second time increment
t_min = t_0;
t_max = t_burn;
delta_t = (t_max - t_min)/N_steps;
t_vals = [0 : N_steps] * delta_t;
% The q and q_dot values at each instant are saved as columns
%      in rectangular arrays.
%%%%%% Baseline model %%%%%%
q_vals = zeros(1, N_steps + 1); % Dedicate memory space
q_dot_vals = zeros(1, N_steps + 1);
% Save starting values
q_vals(:, 1) = y_0;
q_dot_vals(:, 1) = v_0;
Z_prev = [y_0;  v_0];
% Begin time stepping
for m = 2 : N_steps + 1
    % Use @(t, Z) to cause ode45 to treat G_vec as a function of t and Z only.
    %  Error tolerances are not in the list of inputs, so default values are used.
    [t_ode, Z_ode] = ode45(@(t, Z) Falcon_Z_dot(Z, t, m_total, ...
        m_dot, C_d, 0), [t_vals(m - 1)  t_vals(m)], Z_prev);
    % An inconsistency of ode45 is that Z_prev should be a column vector,
    %      but the output array Z_ode consists of rows of Z
    %      corresponding to the instants contained in t_ode.
    % The response at time t_vals(m) is the last row of Z_ode.
    num_evals = length(t_ode);
    % Z for next step is the same as Z at t(m).
    Z_prev = Z_ode(num_evals, :)'; % Convert from row to column storage.
    q_vals(1, m) = Z_prev(1);
    q_dot_vals(1, m) = Z_prev(2);
end
max_y(1) = max(q_vals);
max_v(1) = max(q_dot_vals);

y_baseline = q_vals;
v_baseline = q_dot_vals;

%%%%%% Constant mass model %%%%%%
q_vals = zeros(1, N_steps + 1); % Dedicate memory space
q_dot_vals = zeros(1, N_steps + 1);
% Save starting values
q_vals(:, 1) = y_0;
q_dot_vals(:, 1) = v_0;
Z_prev = [y_0;  v_0];
% Begin time stepping
for m = 2 : N_steps + 1
    % Use @(t, Z) to cause ode45 to treat G_vec as a function of t and Z only.
    %  Error tolerances are not in the list of inputs, so default values are used.
    %  Set m_dot = 0 in computation of dZ/dt
    [t_ode, Z_ode] = ode45(@(t, Z) Falcon_Z_dot(Z, t, m_total,...
        0, C_d, 0), [t_vals(m - 1)  t_vals(m)], Z_prev);
    % An inconsistency of ode45 is that Z_prev should be a column vector,
    %      but the output array Z_ode consists of rows of Z
    %      corresponding to the instants contained in t_ode.
    % The response at time t_vals(m) is the last row of Z_ode.
    num_evals = length(t_ode);
    % Z for next step is the same as Z at t(m).
    Z_prev = Z_ode(num_evals, :)'; % Convert from row to column storage.
    q_vals(1, m) = Z_prev(1);
    q_dot_vals(1, m) = Z_prev(2);
end
max_y(2) = max(q_vals);
max_v(2) = max(q_dot_vals);

y_const_mass = q_vals;
v_const_mass  = q_dot_vals;

%%%%%% No_drag model %%%%%%
q_vals = zeros(1, N_steps + 1); % Dedicate memory space
q_dot_vals = zeros(1, N_steps + 1);
% Save starting values
q_vals(:, 1) = y_0;
q_dot_vals(:, 1) = v_0;
Z_prev = [y_0;  v_0];
% Begin time stepping
for m = 2 : N_steps + 1
    % Use @(t, Z) to cause ode45 to treat G_vec as a function of t and Z only.
    %  Error tolerances are not in the list of inputs, so default values are used.
    %  Set m_dot = 0 in computation of dZ/dt
    [t_ode, Z_ode] = ode45(@(t, Z) Falcon_Z_dot(Z, t, m_total, ...
         m_dot, 0, 0), [t_vals(m - 1)  t_vals(m)], Z_prev);
    % An inconsistency of ode45 is that Z_prev should be a column vector,
    %      but the output array Z_ode consists of rows of Z
    %      corresponding to the instants contained in t_ode.
    % The response at time t_vals(m) is the last row of Z_ode.
    num_evals = length(t_ode);
    % Z for next step is the same as Z at t(m).
    Z_prev = Z_ode(num_evals, :)'; % Convert from row to column storage.
    q_vals(1, m) = Z_prev(1);
    q_dot_vals(1, m) = Z_prev(2);
end
max_y(3) = max(q_vals);
max_v(3) = max(q_dot_vals);

y_No_drag = q_vals;
v_No_drag = q_dot_vals;

%%%%%% Const_grav model %%%%%%
q_vals = zeros(1, N_steps + 1); % Dedicate memory space
q_dot_vals = zeros(1, N_steps + 1);
% Save starting values
q_vals(:, 1) = y_0;
q_dot_vals(:, 1) = v_0;
Z_prev = [y_0;  v_0];
% Begin time stepping
for m = 2 : N_steps + 1
    % Use @(t, Z) to cause ode45 to treat G_vec as a function of t and Z only.
    %  Error tolerances are not in the list of inputs, so default values are used.
    %  Set m_dot = 0 in computation of dZ/dt
    [t_ode, Z_ode] = ode45(@(t, Z) Falcon_Z_dot(Z, t, m_total, ...
        m_dot, C_d, 9.807), [t_vals(m - 1)  t_vals(m)], Z_prev);
    % An inconsistency of ode45 is that Z_prev should be a column vector,
    %      but the output array Z_ode consists of rows of Z
    %      corresponding to the instants contained in t_ode.
    % The response at time t_vals(m) is the last row of Z_ode.
    num_evals = length(t_ode);
    % Z for next step is the same as Z at t(m).
    Z_prev = Z_ode(num_evals, :)'; % Convert from row to column storage.
    q_vals(1, m) = Z_prev(1);
    q_dot_vals(1, m) = Z_prev(2);
end
max_y(4) = max(q_vals);
max_v(4) = max(q_dot_vals);

y_Const_grav = q_vals;
v_Const_grav = q_dot_vals;
disp(' ')
disp(' ')
disp(['Max altitudes (km): ', num2str(max_y/1000)])
disp(['Max speeds (m/s): ', num2str(max_v)])

figure(100)

subplot(2, 2, 1)
plot(t_vals, y_baseline/1000, '-r', ...
    t_vals, y_const_mass/1000, '-b', ...
    t_vals, y_No_drag/1000, '-g', ...
    t_vals, y_Const_grav/1000, '-k', 'linewidth', 0.75)
xlim([0, t_max])
ylabel('Altitude (km)')
legend('Baseline', 'Constant mass', 'No resistance', 'Constant gravity')
subplot(2, 2, 2)
plot(t_vals, v_baseline, '-r', t_vals, v_const_mass, '-b', ...
    t_vals, v_No_drag, '-g', t_vals, v_Const_grav, '-k')
xlim([0, t_max])
xlabel('Time (seconds)')
ylabel('Speed v (m/s)')
