% Asym_free_rotation_stability
% Animates the free rotation of a body about each of its principal axes,
% showing which results in stable rotation and which do not.
% Rotation is always about the Z-axis (with a small perturbation), so the
% initial orientation of the box is varied from case to case.
clear all
close all
I_1 = 1;
I_2 = 4;
I_3 = 5;
I_cases(1, :) = [I_1  I_2  I_3];
I_cases(2, :) = [I_1  I_3  I_2];
I_cases(3, :) = [I_3  I_2  I_1];
case_names={'Rotation about axis with smallest moment of inertia.';
    'Rotation about axis with intermediate moment of inertia.';
    'Rotation about axis with largest moment of inertia.'};
sides = sqrt(I_cases);
L = 2;
% Align horizontal and vertical initially
R_0 = eye(3); % Align initial xyz with global horizontal and vertical
k_0_vec = R_0(3, :)';
omega_0 = [0.001,  0.001, 5]'; % Components relative to initial xyz
% omega_0 = [0.01,  0.01,  5]'; % Components relative to initial xyz
for nn = 1 : 3
    I = diag(I_cases(nn, :));
    disp(['Case #', num2str(nn), ' I = ', num2str(I_cases(nn, :))])
    disp(case_names{nn})
    a = sides(nn, 1);
    b = sides(nn, 2);
    c = sides(nn, 3);
    
    H_G_vec = I * omega_0;
    H_G_mag = norm(H_G_vec);
    
    % Evaluate direction cosines of XYZ relative to intial xyz
    K_vec = H_G_vec/H_G_mag;
    u = cross(K_vec, k_0_vec);
    J_vec = u/norm(u);
    % I_vec = cross(J_vec, K_vec);
    
    % Set initial angles
    theta_0 = acos(K_vec(3));
    phi_0 = acos(J_vec(2));
    psi_0 = 0;
    
    % Estimate time step based on period of overall rotation
    T_average = 2 * pi/norm(omega_0);
    delta_t = T_average/100;
    T_max = 10 * T_average;
    N_vals = ceil(T_max/delta_t);
    angles = zeros(N_vals + 1, 3);
%     t_max = 15;
%     delta_t = t_max/N_vals;
    
    % Initialize ODE
    angles_0 = [psi_0,  theta_0, phi_0]; % Initialize
    
    % Integrate ODEs
    % Set error tolerances
    opts = odeset('RelTol',1e-5,'AbsTol',1e-5);
    [t_vals, angles] = ode45(@(t, angle) F_asym_rot(t, angle, I, H_G_mag),...
        [0 : N_vals] * delta_t, angles_0, opts);
    psi = angles(:, 1);
    theta = angles(:, 2);
    phi = angles(:, 3);
    % Solution of equations of motion ends here
    
    % Check rotational KE and H_G
    psi_dot = H_G_mag * (cos(phi).^2/I(1, 1) + sin(phi).^2/I(2, 2));
    theta_dot = H_G_mag * 0.5 * (1/I(2, 2) - 1/I(1, 1)) * sin(theta) .* sin(2 * phi);
    phi_dot = H_G_mag * (1/I(3, 3) - cos(phi).^2/I(1, 1) - sin(phi).^2/I(2, 2)) .* cos(theta);
    omega_x = -psi_dot .* sin(theta) .* cos(phi) + theta_dot .* sin(phi);
    omega_y = psi_dot .* sin(theta) .* sin(phi) + theta_dot .* cos(phi);
    omega_z = psi_dot .* cos(theta) + phi_dot;
    H_G_vec_all = [omega_x, omega_y omega_z] * I;
    psi_all(:, nn) = psi;
    theta_all(:, nn) = theta;
    phi_all(:, nn) = phi;
    psi_dot_all(:, nn) = psi_dot;
    theta_dot_all(:, nn) = theta_dot;
    phi_dot_all(:, nn) = phi_dot;
    omega_x_all(:, nn) = omega_x;
    omega_y_all(:, nn) = omega_y;
    omega_z_all(:, nn) = omega_z;
    for n = 1 : N_vals + 1
        H_G_mag_check(n) = norm(H_G_vec_all(n, :));
    end
    err_H_g = max(abs(H_G_mag_check - H_G_mag_check(1)))
    KE = 0.5 * (I(1, 1) * omega_x.^2 + I(2, 2) * omega_y.^2 + I(3, 3) * omega_z.^2);
    err_KE = max(abs(KE - KE(1)))
    
    % Plot Eulerian angles and rates
    figure(nn * 10)
    skip = floor(N_vals/30);
    subplot(2, 1, 1)
    hold on
    deg_fac = 180/pi;
    plot(t_vals, psi * deg_fac, '--k', 'linewidth', 0.75)
    plot(t_vals, theta * deg_fac, '-r', 'linewidth', 0.75)
    plot(t_vals(1 : 3 : N_vals + 1), phi(1 : 3 : N_vals + 1) * deg_fac,...
        ':b', 'linewidth', 1.75)
    y_least = 360 * floor(min([min(psi), min(phi), -pi]) * deg_fac/360);
    y_most = 360 * ceil(max(max([psi, phi])) * deg_fac/360);
    ylim([y_least, y_most])
    %yticks([y_least : 360 : y_most])
    xlabel('Time (s)')
    ylabel('Angle (deg)')
    legend('\psi', '\theta', '\phi')
    title(case_names{nn})
    hold off
    subplot(2, 1, 2)
    hold on
    plot(t_vals, psi_dot, '--k', 'linewidth', 0.75)
    plot(t_vals, theta_dot, '-r', 'linewidth', 0.75)
    plot(t_vals(1 : 3 : N_vals + 1), phi_dot(1 : 3 : N_vals + 1),...
        ':b', 'linewidth', 1.75)
    xlabel('Time (s)')
    ylabel('Angular velocity (rad/s)')
    legend('d\psi/dt', 'd\theta/dt', 'd\phi/dt')
    y_least = floor(min([min(psi_dot), min(phi_dot), -10]/10) * 10);
    y_most = ceil(max(max([psi_dot, phi_dot]))/10) * 10;
    ylim([y_least, y_most])
%     saveas(gcf, 'Angles vs time.pdf', 'pdf')
    hold off
end

figure(50)
subplot(3, 1, 1)
plot(t_vals, theta_all(:, 1) * 180/pi, 'b--', t_vals, theta_all(:, 2) * 180/pi, 'r-',...
    t_vals, theta_all(:, 3) * 180/pi, 'g:')
xlim([0, max(t_vals)]); xlabel('Time (s)');
ylim([-5, 185]); ylabel('d\theta/dt')
yticks([0 : 45 : 180])
legend('Case 1','Case 2','Case 3')

Omega_all = theta_dot_all; + phi_dot_all;
subplot(3, 1, 2)
plot(t_vals, psi_dot_all(:, 1), 'b-', t_vals, phi_dot_all(:, 1), 'b-',...
    t_vals, psi_dot_all(:, 2), 'r-', t_vals, phi_dot_all(:, 2), 'r-', ...
    t_vals, psi_dot_all(:, 3), 'g--', t_vals, phi_dot_all(:, 3), 'g--',...
    'linewidth', 1, 'MarkerSize', 1)
xlim([0, max(t_vals)]); xlabel('Time (s)');
ylabel('d\psi/dt')

subplot(3, 1, 3)
plot(t_vals, omega_z_all(:, 1), 'b-',...
    t_vals, omega_z_all(:, 2), 'r-', ...
    t_vals, omega_z_all(:, 3), 'g--', 'linewidth', 1)
xlim([0, max(t_vals)]); xlabel('Time (s)');
ylabel('d\omega_z/dt')

return
%% Run this Section to animate some of the solutions
% Select the case of interest:
nn=2; % 1, 2, 3

% Much of the code below is repeated from above, to reassign the various
% variables.
I = diag(I_cases(nn, :));
disp(['Case #', num2str(nn), ' I = ', num2str(I_cases(nn, :))])
disp(case_names{nn})
a = sides(nn, 1);
b = sides(nn, 2);
c = sides(nn, 3);

H_G_vec = I * omega_0;
H_G_mag = norm(H_G_vec);

% Evaluate direction cosines of XYZ relative to intial xyz
K_vec = H_G_vec/H_G_mag;
u = cross(K_vec, k_0_vec);
J_vec = u/norm(u);
% I_vec = cross(J_vec, K_vec);

% Set initial angles
theta_0 = acos(K_vec(3));
phi_0 = acos(J_vec(2));
psi_0 = 0;

% Estimate time step based on period of overall rotation
T_average = 2 * pi/norm(omega_0);
delta_t = T_average/100;
% T_max = 4 * T_average;
T_max = 20 * T_average;
N_vals = ceil(T_max/delta_t);
angles = zeros(N_vals + 1, 3);
%     t_max = 15;
%     delta_t = t_max/N_vals;

% Initialize ODE
angles_0 = [psi_0,  theta_0, phi_0]; % Initialize

% Integrate ODEs
% Set error tolerances
opts = odeset('RelTol',1e-5,'AbsTol',1e-5);
[t_vals, angles] = ode45(@(t, angle) F_asym_rot(t, angle, I, H_G_mag),...
    [0 : N_vals] * delta_t, angles_0, opts);
psi = angles(:, 1);
theta = angles(:, 2);
phi = angles(:, 3);
% Solution of equations of motion ends here

% Check rotational KE and H_G
psi_dot = H_G_mag * (cos(phi).^2/I(1, 1) + sin(phi).^2/I(2, 2));
theta_dot = H_G_mag * 0.5 * (1/I(2, 2) - 1/I(1, 1)) * sin(theta) .* sin(2 * phi);
phi_dot = H_G_mag * (1/I(3, 3) - cos(phi).^2/I(1, 1) - sin(phi).^2/I(2, 2)) .* cos(theta);
omega_x = -psi_dot .* sin(theta) .* cos(phi) + theta_dot .* sin(phi);
omega_y = psi_dot .* sin(theta) .* sin(phi) + theta_dot .* cos(phi);
omega_z = psi_dot .* cos(theta) + phi_dot;
H_G_vec_all = [omega_x, omega_y omega_z] * I;

%%%%%%%%% Begin animations %%%%%%%%%%%%%
% Transformation from XYZ to perceptual XYZ
RR = R_0' * S_transf(psi_0, theta_0, phi_0);
% Define xyz coordinates of points at corners of the box
xyz_points(:, 1) = [-a, -b, -c]'; % (-a,-b,-c)
xyz_points(:, 2) = [+a, -b, -c]'; % (+a,-b,-c)
xyz_points(:, 3) = [-a, +b, -c]'; % (-a,+b,-c)
xyz_points(:, 4) = [+a, +b, -c]'; % (+a,+b,-c)
xyz_points(:, 5) = [-a, -b, +c]'; % (-a,-b,+c)
xyz_points(:, 6) = [+a, -b, +c]'; % (+a,-b,+c)
xyz_points(:, 7) = [-a, +b, +c]'; % (-a,+b,+c)
xyz_points(:, 8) = [+a, +b, +c]'; % (+a,+b,+c)
% Define xyz coordinates of axes that emerge from each face
x_axis = [[a, 0,  0]; [(a + 0.6), 0, 0]]';
y_axis = [[0, b,  0]; [0, (b + 0.6), 0]]';
z_axis = [[0, 0,  c]; [0, 0, (c + 0.6)]]';
axes_xyz = [x_axis, y_axis, z_axis];

num_plots = N_vals + 1;
f = figure(100);
% f.WindowState = 'maximized';

% Skip some instants when animating to make it go more quickly.
n_skip = 10;
for n = 1 : n_skip : num_plots
    hold off
    newplot
    hold on
    view([135, 20])
    % Evaluate tranformation from xyz to global XYZ
    R_to_ref = RR * S_transf(psi(n), theta(n), phi(n))';
    % Draw H_G
    H_G_XYZ = R_to_ref * H_G_vec_all(n, :)';
    e_H_G = H_G_XYZ/norm(H_G_XYZ);
    
    p = quiver3(0, 0, 0, e_H_G(1)* 1.3 * L, e_H_G(2)* 1.3 * L, ...
        e_H_G(3)* 1.3 * L);
    p.LineWidth = 2;
    p.Color = [0, 0.5, 0.5];
    p.AutoScale = 'off';
    p.MaxHeadSize = 0.5;
    text(K_vec(1)* 1.35 * L, K_vec(2)* 1.35 * L, K_vec(3)* 1.35 * L, 'H_G')
    
    % Draw global X, Y, and Z axes
    plot3([-1.2 * L, 1.2 * L], [0, 0], [0, 0], ':k', 'linewidth', 1.0)
    text( 1.3 * L, 0, 0, 'X', 'FontWeight','bold')
    plot3([0, 0], [-1.2 * L, 1.2 * L], [0, 0], ':k', 'linewidth', 1.0)
    text(0, 1.3 * L, 0, 'Y', 'FontWeight','bold')
    plot3([0, 0], [0, 0], [-1.2 * L, 1.2 * L], ':k', 'linewidth', 1.0)
    text(0, 0, 1.3 * L, 'Z', 'FontWeight','bold')
    
    % Compute fixed coordinates of corners
    R_to_ref = RR * S_transf(psi(n), theta(n), phi(n))';
    XYZ_pts = R_to_ref * xyz_points;
    axes_XYZ =  R_to_ref * axes_xyz;
    % Draw boz faces
    Rect_face(XYZ_pts, [1, 2, 4, 3], [0.4, 0, 0])
    Rect_face(XYZ_pts, [1, 2, 6, 5], [0, 0, 0.4])
    Rect_face(XYZ_pts, [1, 3, 7, 5], [0, 0.4, 0])
    Rect_face(XYZ_pts, [5, 6, 8, 7], [1, 0, 0])
    Rect_face(XYZ_pts, [3, 4, 8, 7], [0, 0, 1])
    Rect_face(XYZ_pts, [2, 4, 8, 6], [0, 1, 0])
    
    % Draw xyz axes coming out of the box and add label
    plot3(axes_XYZ(1, 1:2), axes_XYZ(2, 1:2), axes_XYZ(3, 1:2), '-k', 'linewidth', 1)
    text(axes_XYZ(1, 2) * 1.1, axes_XYZ(2, 2) * 1.1, axes_XYZ(3, 2) * 1.1, 'x')
    plot3(axes_XYZ(1, 3:4), axes_XYZ(2, 3:4), axes_XYZ(3, 3:4), '-k', 'linewidth', 1)
    text(axes_XYZ(1,4) * 1.1, axes_XYZ(2, 4) * 1.1, axes_XYZ(3, 4) * 1.1, 'y')
    plot3(axes_XYZ(1, 5:6), axes_XYZ(2, 5:6), axes_XYZ(3, 5:6), '-k', 'linewidth', 1)
    text(axes_XYZ(1, 6) * 1.1, axes_XYZ(2, 6) * 1.1, axes_XYZ(3, 6) * 1.1, 'z')
    
    % Draw angular velocity direction
    omega_n_xyz = [omega_x(n), omega_y(n), omega_z(n)]';
    % Transform to fixed XYZ comps
    omega_n = R_to_ref * omega_n_xyz;
    e_omega = 1.1 * L * omega_n/norm(omega_n);
    q = quiver3(0, 0, 0, e_omega(1), e_omega(2),e_omega(3));
    q.LineWidth = 1.5;
    q.MaxHeadSize = 1;
    q.Color = [0.9, 0.4 1];
    q.AutoScale = 'off';
    p.LineWidth = 2;
    text(1.15 * e_omega(1), 1.15* e_omega(2), 1.15 * e_omega(3), ...
        'e_\omega', 'FontSize', 12)
    %     Project e_omega onto fixed XY plane
    plot3([e_omega(1), e_omega(1)], [e_omega(2), e_omega(2)], [0, e_omega(3)], ...
        '--k', 'linewidth', 1) % Projection line
    plot3([0, e_omega(1)], [0, e_omega(2)], [e_omega(3), e_omega(3)],...
        '--k','linewidth', 1) % Transverse to Z in to head of e_omega
    plot3([0, e_omega(1)], [0, e_omega(2)], [0, 0],...
        '--k','linewidth', 1) % Transverse to Z in XY plane
    
    ax_len = 1.25 * norm([a, b, c]);
    axis equal
    %     axis off
    grid on
    axis([-ax_len  ax_len  -ax_len  ax_len  -ax_len  ax_len])
    title(['H_G = [', num2str(H_G_vec'), ']'])
    text(0.5 * ax_len, 0.5 * ax_len, -0.9 * ax_len, ['t(', num2str(n), ')=', num2str(t_vals(n))]);
    F(n)=getframe;
    if n > 1
        pause(delta_t);
    else
        pause(2)
    end
end

% Now to play back the movie again use:
% movie(F,1,30); % Play once at 30fps.  Note, doesn't work in 2024b