% Trajectory_with_air_resistance.m
% Solves for the response of a projectile (golf ball) in free flight,
% subject to aerodynamic drag of the form f=c*v^2.  The launch angle of the
% ball is varied and results are plotted for several trajectories.

clear all; close all

% Define system parameters here:
g = 9.807;
v_term = 32; % Terminal Velocity in m/s
k_m = g/v_term^2;
% Set initial conditions:
N = 2;
% ode45 can decide the instants at which to output the response.
%      Instead, the procedure implemented here uses a fixed time step.
N_steps =  100; % Number time steps for response output
t_min = 0;
t_max = 15;
delta_t = (t_max - t_min)/N_steps;
figure(100)
hold on
% Set initial condition cases
v_0 = 80;
psi_vals = [60 45  30 15] * pi/180; % Initial angle of elevation
N_case = length(psi_vals); 
for nn = 1 : N_case % loop over each initial angle
    psi = psi_vals(nn);
    % Initial Position (x,y,z)
    q_0 = [0;  0];  % Both q_0 and q_dot_0 should be column vectors.
    % Initial Velocities
    q_dot_0 = v_0 *[cos(psi);  sin(psi)];
    % State Space Initial Condition
    Z_0 = [q_0; q_dot_0];
    t_0 = 0;
    t_vals = [0:N_steps - 1] * delta_t;
    % The q and q_dot values at each instant are saved as columns
    %      in rectangular arrays.
    [t_ode, Z_ode] = ode45(@(t, Z) golf_eom(Z, t, k_m, g), t_vals, Z_0);
    q_vals = Z_ode(:,1:N); % Extract displacements from state space vector
    q_dot_vals = Z_ode(:,N+1:2*N);
    % Extract xyz
    x_coord=q_vals(:,1); % extract x-position
    z_coord=q_vals(:,2); % extract z-position
    % Find instand of impact with the ground
    ind_impact = find(z_coord<0,1)-1; % find the first instant when z becomes 
        % negative, then go back one sample
    
    % Plot the result for each case.
    figure(100)
    subplot(2, 1, 1)
    plot(t_ode(1:ind_impact), x_coord(1:ind_impact))
    xlabel('Time (s)')
    ylabel('x (m)')
    ylim([0  160])
    text(t_ode(floor(ind_impact/2)), x_coord(floor(ind_impact/2)), num2str(psi*180/pi))
    hold on

    subplot(2, 1, 2)
    plot(t_ode(1:ind_impact), z_coord(1:ind_impact))
    ylim([0  150])
    xlabel('Time (s)')
    ylabel('z (m)')
    text(t_ode(floor(ind_impact/2)), z_coord(floor(ind_impact/2)), num2str(psi*180/pi))
    hold on

    figure(110)
    plot(x_coord, z_coord)
    xlabel('x (m)')
    ylabel('z (m)')
    xlim([0  150])
    ylim([0  90])
    text(x_coord(floor(ind_impact/2)), z_coord(floor(ind_impact/2)), num2str(psi*180/pi))
    hold on
end

figure(110)
axis equal
ylim([0 90])

